#!/usr/bin/env node

const crypto = require('crypto');
const fs = require("fs/promises");
const path = require("path");

const selection = require("../source/icons/selection.json");

const buildBaseRules = (hash) => `
@font-face {
  font-family: 'pirati-ui';
  src:  url('../fonts/pirati-ui.eot?${hash}');
  src:  url('../fonts/pirati-ui.eot?${hash}#iefix') format('embedded-opentype'),
    url('../fonts/pirati-ui.ttf?${hash}') format('truetype'),
    url('../fonts/pirati-ui.woff?${hash}') format('woff'),
    url('../fonts/pirati-ui.svg?${hash}#pirati-ui') format('svg');
  font-weight: normal;
  font-style: normal;
  font-display: block;
}

[class^="ico--"], [class*=" ico--"] {
  /* use !important to prevent issues with browser extensions that change fonts */
  font-family: 'pirati-ui' !important;
  speak: never;
  font-style: normal;
  font-weight: normal;
  font-variant: normal;
  text-transform: none;
  line-height: 1;

  /* Better Font Rendering =========== */
  -webkit-font-smoothing: antialiased;
  -moz-osx-font-smoothing: grayscale;
}
`;

const buildIconRules = (icons) => {
  return icons.map(icon => `.${icon.class}:before { content: "${icon.content}"; }`).join("\n");
}

const getHash = (icons) => {
  return crypto.createHash('md5').update(icons, 'utf8').digest('hex');
}

const buildRuleset = (icons) => {
  const iconRules = buildIconRules(icons);
  const hash = getHash(iconRules);
  const baseRules = buildBaseRules(hash);
  return [baseRules, iconRules].join("\n");
}

const buildPattern = (icons) => {
  const body = icons.map(icon => `
<div class="card">
  <div class="flex justify-center items-center h-16 text-2xl border-b border-grey-100 bg-grey-50 p-2">
    <i class="${icon.class}"></i>
  </div>
  <div class="card__body p-2 text-2xs">
    <code>.${icon.class}</code>
  </div>
</div>`).join("\n");
  return `<div class="grid grid-cols-2 sm:grid-cols-4 lg:grid-cols-8 xl:grid-cols-16 gap-2 items-center">${body}</div>`;
}

async function writeFile(path, content) {
  let filehandle;

  try {
    filehandle = await fs.open(path, "w");
    filehandle.write(content);
  } finally {
    if (filehandle !== undefined)
      await filehandle.close();
  }
}


const prefix = selection.preferences.fontPref.prefix;
const icons = selection.icons.map(icon => {
  const content = `${'\\'}${icon.properties.code.toString(16)}`;
  return icon.properties.name.split(", ").map(iconName => ({
    class: `${prefix}${iconName}`,
    content
  }));
}).flat();

// Write PCSS
writeFile(path.join(path.dirname(__filename), "..", "source", "css", "atoms", "icons.pcss"), buildRuleset(icons));

// Write Pattern Lab pattern
writeFile(path.join(path.dirname(__filename), "..", "source", "_patterns", "00-atoms", "02-icons", "icons.mustache"), buildPattern(icons));
