"""
Pirati Authentication
"""

import json
import re
from urllib import request

from celery import shared_task
from django.core.mail import send_mail
from django.conf import settings
from requests_oauthlib import OAuth2Session
from unidecode import unidecode


# some parameters to indicate that status updating is not possible
STATUS_UPDATES = False

# display tweaks
LOGIN_MESSAGE = "Přihlásit se pirátskou identitou"


PIRATI_ENDPOINT_URL = f"{settings.PIRATI_REALM_URL}/protocol/openid-connect/auth"
PIRATI_TOKEN_URL = f"{settings.PIRATI_REALM_URL}/protocol/openid-connect/token"
PIRATI_USERINFO_URL = f"{settings.PIRATI_REALM_URL}/protocol/openid-connect/userinfo"


###############################################################################
# Custom helper functions


def call_octopus_api(query, variables=None):
    payload = json.dumps({"query": query, "variables": variables or {}}).encode("utf-8")
    req = request.Request(settings.OCTOPUS_API_URL, method="POST")
    req.add_header("Content-Type", "application/json; charset=utf-8")
    req.add_header("Authorization", f"Bearer {settings.OCTOPUS_API_TOKEN}")
    response = request.urlopen(req, payload)
    data = json.loads(response.read())
    if "errors" in data:
        raise RuntimeError(
            f"API call failed!\n - query:\n----\n{query}\n----\n - variables:\n----\n{variables}\n----\n- response:\n----\n{data}\n----\n"
        )
    return data


def get_octopus_person(username):
    query = """
    query data ($username: String!) {
        allPeople (first: 1, filters: {username: {iExact: $username}}) {
            edges {
                node {
                    username
                    displayName
                    email
                }
            }
        }
    }
    """
    variables = {"username": username}
    data = call_octopus_api(query, variables)
    return data["data"]["allPeople"]["edges"][0]["node"]


def get_octopus_groups():
    query = """
    query {
        allGroups (filters: {voting: true}) {
            edges {
                node {
                    id
                    name
                }
            }
        }
    }
    """
    data = call_octopus_api(query)
    return [edge["node"] for edge in data["data"]["allGroups"]["edges"]]


def get_octopus_group(group_id):
    query = """
    query data ($id: GlobalID!) {
        group (id: $id) {
            id
            name
        }
    }
    """
    variables = {"id": group_id}
    data = call_octopus_api(query, variables)
    return data["data"]["group"]


def get_octopus_group_members(group_id):
    query = """
    query data ($id: GlobalID!) {
        group (id: $id) {
            id
            memberships {
                person {
                    username
                    displayName
                    email
                }
            }
        }
    }
    """
    variables = {"id": group_id}
    data = call_octopus_api(query, variables)
    return [m["person"] for m in data["data"]["group"]["memberships"]]


def group_sorter(group):
    name = unidecode(group["name"])
    if name == "Celostatni forum":
        return f"0_{name}"

    if re.match(r"^K[FS] ", name):
        return f"1_{name[3:]}"

    if re.match(r"^M[FS] Ch", name):
        return f"3_{name[3:]}"
    if re.match(r"^M[FS] [ABCDEFGH]", name):
        return f"2_{name[3:]}"
    if re.match(r"^M[FS] ", name):
        return f"4_{name[3:]}"

    return f"9_{name}"


def is_old_group(group_id):
    try:
        int(group_id)
        return True
    except:
        pass

    if group_id.startswith("deadbeef-babe-"):
        return True

    return False


def person_to_user_info(person):
    return {
        "type": "pirati",
        "id": person["username"].lower(),
        "name": person["displayName"],
        "info": {"email": person["email"], "name": person["displayName"]},
        "token": {},
    }


def old_member_to_user_info(member):
    return {
        "type": "pirati",
        "id": member["username"].lower(),
        "name": member["username"],
        "info": {"email": member["email"], "name": member["username"]},
        "token": {},
    }


###############################################################################
# Celery tasks


@shared_task(
    autoretry_for=(Exception,),
    max_retries=160,  # 3 days
    retry_backoff=True,
    retry_backoff_max=1800,
)
def send_email_task(recipient, subject, body):
    send_mail(subject, body, settings.SERVER_EMAIL, [recipient], fail_silently=False)


###############################################################################
# Helios stuff

can_list_category_members = True


def get_auth_url(request, redirect_url):
    request.session["pirate_redirect_url"] = redirect_url
    oauth = OAuth2Session(settings.PIRATI_CLIENT_ID, redirect_uri=redirect_url)
    url, state = oauth.authorization_url(PIRATI_ENDPOINT_URL)
    return url


def get_user_info_after_auth(request):
    oauth = OAuth2Session(
        settings.PIRATI_CLIENT_ID, redirect_uri=request.session["pirate_redirect_url"]
    )
    token = oauth.fetch_token(
        PIRATI_TOKEN_URL,
        client_secret=settings.PIRATI_CLIENT_SECRET,
        code=request.GET["code"],
    )
    response = oauth.get(PIRATI_USERINFO_URL)
    data = response.json()
    person = get_octopus_person(data["preferred_username"])
    info = person_to_user_info(person)
    info["user_id"] = info.pop("id")
    return info


def do_logout(user):
    return None


def update_status(token, message):
    pass


def send_message(user_id, user_name, user_info, subject, body):
    recipient = "%s <%s>" % (user_info.get("name", user_name), user_info["email"])
    send_email_task.delay(recipient, subject, body)


def generate_constraint(category_id, user):
    return category_id


def eligibility_category_id(constraint):
    return constraint


def check_constraint(constraint, user):
    if is_old_group(constraint):
        userinfo = json.load(
            request.urlopen("https://graph.pirati.cz/user/" + user.user_id)
        )
        id = userinfo["id"]
        usergroups = json.load(
            request.urlopen("https://graph.pirati.cz/" + id + "/groups")
        )
        for usergroup in usergroups:
            if usergroup["id"] == constraint:
                return True
    else:
        people = get_octopus_group_members(constraint)
        usernames = [person["username"].lower() for person in people]
        if user.user_id in usernames:
            return True
    return False


def list_categories(user):
    return sorted(get_octopus_groups(), key=group_sorter)


def list_category_members(category_id):
    if is_old_group(category_id):
        members = json.load(
            request.urlopen("https://graph.pirati.cz/" + category_id + "/members")
        )
        return [old_member_to_user_info(member) for member in members]
    else:
        people = get_octopus_group_members(category_id)
        return [person_to_user_info(person) for person in people]


def pretty_eligibility(constraint):
    if is_old_group(constraint):
        group = json.load(request.urlopen("https://graph.pirati.cz/" + constraint))
        name = group["username"]
    else:
        group = get_octopus_group(constraint)
        name = group["name"]
    return f"Osoby ve skupině „{name}“"


#
# Election Creation
#


def can_create_election(user_id, user_info):
    return True
