from django.core.management.base import BaseCommand
from contracts.models import Signee
import requests


class Command(BaseCommand):
    help = "Update signee information from IČO numbers."

    def handle(self, *args, **options) -> None:
        do_all = False

        for signee in Signee.objects.filter(ico_number__isnull=False):
            try:
                ares_info = requests.get(
                    f"https://ares.gov.cz/ekonomicke-subjekty-v-be/rest/ekonomicke-subjekty/{signee.ico_number}"
                )

                ares_info.raise_for_status()

                formatted_ares_data = ares_info.json()

                street = ""
                
                if "nazevUlice" in formatted_ares_data["sidlo"]:
                    street += formatted_ares_data["sidlo"]["nazevUlice"]

                if "cisloDomovni" in formatted_ares_data["sidlo"]:
                    street += (" " + str(formatted_ares_data["sidlo"]["cisloDomovni"]))

                if "cisloOrientacni" in formatted_ares_data["sidlo"]:
                    street += ("/" + str(formatted_ares_data["sidlo"]["cisloOrientacni"]))

                changed = False

                attribute_sets = {
                    "name": formatted_ares_data["obchodniJmeno"],
                    "address_street_with_number": street,
                    "address_district": formatted_ares_data["sidlo"]["nazevCastiObce"],
                    "address_zip": formatted_ares_data["sidlo"]["psc"],
                    "address_country": formatted_ares_data["sidlo"]["nazevStatu"]
                }

                for attribute, value in attribute_sets.items():
                    if str(getattr(signee, attribute)) != str(value):
                        changed = True
                        self.stdout.write(f"\nUpdating {signee}'s {attribute}:\n\tOLD:\t{getattr(signee, attribute)}\n\tNEW:\t{value}\n")

                if changed:
                    if not do_all:
                        change_opt = input("\nConfirm? [y/n/selective/all]: ")

                        while change_opt not in ("y", "selective", "n", "all"):
                            change_opt = input("\nConfirm? [y/n/selective/all]: ")

                        if change_opt not in ("y", "selective"):
                            if change_opt == "all":
                                do_all = True

                            continue

                        attrs_to_change = []

                        if change_opt == "y":
                            attrs_to_change = attribute_sets.keys()
                        else:
                            attrs_to_change = input(
                                "Which attrs? [name, address_street_with_number, address_district, "
                                "address_zip, address_country - separated by commas]: "
                            ).split(",")
                    else:
                        attrs_to_change = attribute_sets.keys()

                    changed_attrs = []

                    for attribute, value in attribute_sets.items():
                        if attribute in attrs_to_change:
                            changed_attrs.append(attribute)

                            setattr(signee, attribute, value)

                    signee.save()

                    self.stdout.write(self.style.SUCCESS(f"\nChanged ({','.join(changed_attrs)})."))
                    self.stdout.write("\n\n\n")

            except Exception:
                self.stdout.write(self.style.WARNING(f"Signee {signee}'s IČO appears to be invalid."))