import io
import os
import re
import shutil
import string
from datetime import date, datetime

import yaml
from django.conf import settings
from django.core.files import File
from django.core.management.base import BaseCommand
from django.db import models
from git import Repo
from postal.parser import parse_address

from ...models import (
    Contract,
    Contractee,
    ContracteeSignature,
    ContracteeSignatureRepresentative,
    ContractFile,
    ContractFilingArea,
    ContractIssue,
    ContractType,
    Signee,
    SigneeSignature,
    SigneeSignatureRepresentative,
)


class Command(BaseCommand):
    help = "Sync contract data from a remote Git repository."

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.normal_import_count = 0
        self.partial_import_count = 0
        self.already_imported_count = 0
        self.normalization_count = 0
        self.issue_count = 0
        self.fatal_error_count = 0

    def add_arguments(self, parser) -> None:
        parser.add_argument(
            "repo_url",
            type=str,
            help="URL of the Git repository to clone",
        )
        parser.add_argument(
            "branch",
            type=str,
            help="Branch to use",
        )
        parser.add_argument(
            "--directory",
            type=str,
            help="Directory to store the cloned repository in",
        )
        parser.add_argument(
            "--delete",
            action="store_true",
            help="Delete the old temporary storage directory, if it exists.",
        )
        parser.add_argument(
            "--existing",
            action="store_true",
            help="Use the existing storage directory, as long as it exists.",
        )
        parser.add_argument(
            "--purge",
            action="store_true",
            help="Purge all previous contracts, types and filing areas before the import.",
        )

    def use_issue(self, name: str) -> None:
        issues = ContractIssue.objects.filter(name=name).all()

        if len(issues) != 0:
            issue = issues[0]
        elif len(issues) > 1:
            for issue in issues[1:]:
                for contract in issue.contracts:
                    contract.issues.add(issue)

                issue.delete()
        else:
            issue = ContractIssue(name=name)
            issue.save()

        return issue

    def normalize_type(self, type_name: str) -> str:
        type_name = string.capwords(type_name)

        patterns = (
            (r"\s\s+", " "),
            (r" O ", " o "),
            (r" S ", " s "),
            (r" K ", " k "),
            (r" V ", " v "),
            (r"Nda", "NDA"),
            (r"Dds", "DDS"),
            (r"^Dohoda o Podpoře.*$", "Dohoda o Podpoře"),
            (
                r"^(Dohoda o Ochraně Důvěrných Informací \(nda\)|Dohoda o Ochraně Důvě"
                r"rných Dat|Dohoda o Ochraně Důvěrných Informací|Smlouva o Zachování D"
                r"ůvěrnosti Informací|Smlouva o Zachování Důvěrnosti Informací \(nda\)"
                ")$",
                "NDA",
            ),
            (r"^(Darovací.*|Ds)$", "Darovací Smlouva"),
            (r".*Dodatek.*", "Dodatek"),
            (
                r"^(Dpp|Dohoda o Provední Práce|Dohoda o Proveení Práce)$",
                "Dohoda o Provedení Práce",
            ),
            (r"^Dpč$", "Dohoda o Pracovní Činnosti"),
            (r"^NS$", "Nájemní Smlouva"),
            (r"^Koaliční Smlouva.*$", "Koaliční Smlouva"),
            (r"^Pronajem$", "Pronájem"),
            (
                r"^(Rácová Smlouva|Rámcová Kupní Smlouva|Rámcová Smlouva|Rámcová Smlouva- Do"
                r"datek|Ramcova Smlouva o Najmu Kopií Filmů A Rámcová Smlouva Podlicenční|Rs"
                r")$",
                "Rámcová Smlouva",
            ),
            (
                r"^(Oof|Oosf|Osf|Potvrzení o Funkci|Potvrzeni o Stranicke Funkci|Potvrzení o"
                r"Stranické Funkci)",
                "Osvědčení o Stranické Funkci",
            ),
            (r"^Smlouva o Bezúročné Zápůjčce.*$", "Bezúročná Zápůjčka Peněz"),
            (
                r"^(Smlouva o Dilo|Smlouva o Dílo A Veřejná Licenční Smlouva|Smlouva o Dílo - "
                r"Dodatek|Sod)$",
                "Smlouva o Dílo",
            ),
            (r"^Smlouva o Nájmu.*$", "Smlouva o Nájmu"),
            (
                r"^Smlouva o Poskytovani Pravnich Sluzeb$",
                "Smlouva o Poskytování Právních Služeb",
            ),
            ("\.$", ""),
            (r"^Ks$", "Koaliční Smlouva"),
            (r"^Pm$", "Plná moc"),
            (r"Pronajmu", "Pronájmu"),
            (r"^(Zl .*|(Zadavací|Zdávací) list)$", "Zadávací list"),
            (r"Hpp", "Hlavní Pracovní Poměr"),
            (r"^Sops - Smlouva o Poskytnutí Služeb", "Smlouva o Poskytnutí Služeb"),
            (r"^Suv$", "Smlouva o Uměleckém Vystoupení"),
            (
                r"^(Příkaz|Příkazní|Prikazni Smlouva|Příkazní Smlouva|Příkazní Smlouva, Bude"
                r"Fakturováno|Příkazní Smlouva (dodatek)|Příkazní Smlouva - Dodatek|Příkazní"
                r"Smlouva Dodatek|Příkazní Smlouva- Dodatek|Příkazní Smlouva - Dodatek Č\. 1"
                r"|Příkazní Smlouva - Dodatek Č\.1|Příkazní Smlouva Dodatek Č\. 1|Příkazní S"
                r"mlouva Dodatek Č\.1|Příkazní Smlouva s Dodavatelem|Příkazní Smlouva- Ukonč"
                r"ení|Příkazní Smlouvy|Příkazní Smlova|Příkazní Smluva|Příkazní Smuva|Příkaz"
                r"ní Spisovna|Příkzaní Smlouva|Ps)$",
                "Příkazní smlouva",
            ),
        )

        for pattern in patterns:
            type_name = re.sub(pattern[0], pattern[1], type_name)

        return type_name

    def normalize_department(self, type_name: str) -> str:
        type_name = type_name.strip()

        patterns = (
            (r"\s\s+", " "),
            (r"^Kraské sdružení Praha$", "Krajské sdružení Praha"),
            (r"^republikové předsednictvo$", "Republikové předsednictvo"),
            (r"^KS ", "Krajské sdružení "),
            (r"^(MS |místní sdružení )", "Místní sdružení "),
            (r"^(Ústecký kraj|Ustecký kraj)$", "Krajské sdružení Ústecký kraj"),
            (
                r"^Moravskoslezský kraj$",
                "Krajské sdružení Moravskoslezský kraj",
            ),
            (
                r"^Karlovarský kraj$",
                "Krajské sdružení Karlovarský kraj",
            ),
            (
                r"Jihočeská kraj",
                "Jihočeský kraj",
            ),
            (r"^(Krajského |krajské |Kajské )", "Krajské "),
            ("^Poslanecký klub České pirátské strany$", "Poslanecký klub"),
            (r"Středočeký", "Středočeský"),
            (r"^Zahraničního odboru$", "Zahraniční odbor"),
            (r"JčK", "Jihočeský kraj"),
            (r"SčK", "Středočeský kraj"),
            (r"(ÚsK|UsK)", "Ústecký kraj"),
            (r"JmK", "Jihomoravský kraj"),
            (r"PaK", "Pardubický kraj"),
            (r"KhK", "Královehradecký kraj"),
            (r"(Prsonální|personální)", "Personální"),
            (r"^administrativní ", "Administrativní "),
            (r"technický", "Technický"),
            (r"Mediálni", "Technický"),
            (r"^řešitel ", "Řešitel "),
            (r"^předsednictvo ", "Předsednictvo "),
            (r"olomoucký", "Olomoucký"),
            (r"^místní ", "Místní "),
            (r"^celostátní ", "Celostátní "),
            (r"odbor", "Odbor"),
            (r"PKS", "Předsednictvo krajského sdružení"),
            (r"( KS | Krajského sdružení )", " krajského sdružení "),
            (
                r"^(Předsednictvo krajského sdružení |Předsednictvo |Místní předsednictvo )",
                "",
            ),
            (r"^Krajské předsednictvo ", "Krajské sdružení "),
            (r"ého kraje$", "ý kraj"),
            (r"^Olomouc$", "Místní sdružení Olomouc"),
            (r"^Olomoucký kraj$", "Krajské sdružení Olomoucký kraj"),
            (r"^Pardubický kraj$", "Krajské sdružení Pardubický kraj"),
            (r"^Jihočeský kraj$", "Krajské sdružení Jihočeský kraj"),
            (r"^Královehradecký kraj$", "Krajské sdružení Královehradecký kraj"),
            (r"^Pardubický kraj$", "Krajské sdružení Pardubický kraj"),
        )

        for pattern in patterns:
            type_name = re.sub(pattern[0], pattern[1], type_name)

        return type_name

    def normalize_filing_area(self, area_name: str) -> str:
        area_name = string.capwords(area_name)

        patterns = (
            (r"\s\s+", " "),
            (
                r"^(Cenrální Spisovna|Censtrální Spisovna|Centrála|Centrálách Archiv Str"
                r"any|Centrála Strany|Centrální Achiv Strany|Centrální Archiv|Centralni "
                r"Archiv Strany|Centrální Archiv Strany|Centrální Skupina|Centrální Spid"
                r"ovna|Centrální Spisivna|Centrální Spisovna|Centrální Spísovna|Centráln"
                r"í Spisovna Strany|Centrální Sposovna|Centrální Sposovná|Centrální Úlož"
                r"na|Centrální Úschovna|Česká Pirítská Strana, Na Moráni 360\/3, Praha 2"
                r"|Cnterální Spisovna|Archiv|Archív|Archiv Centrály|Archiv České Pirátsk"
                r"é Strany|Archiv Pice|Archív Piráti|Archiv Pirátské Strany|Archiv Praha"
                r"|Archiv, Sídlo Strany|Archiv, Sídlo Strany,|Archív, Sídlo Strany|Archi"
                r"v, Sídlo Strany, Řehořova 19, Praha 3|Archiv, Sídlo Strany, Řehořova 1"
                r"9, Praha 3, Zastupitelský Klub Pirátů, Mariánské Nám. 2, Praha 1 \/pří"
                r"lohy|Archiv, Sídlo Strany, Řehořova 943\/19, Praha 3|Archív, Sídlo Str"
                r"any, Řehořova 943\/19, Praha 3|Archiv, Sídlo Strany, Řehova 19, Praha "
                r"3|Archiv Služeb|Archiv Smlouvy|Archiv Starny|Archiv Strany|Archív Stra"
                r"ny|Archiv Strany, Řehořova 943\/19, Praha|Archiv Strany, Sídlo Strany|"
                r"Archiv Strany \(zatím Koks, Do Konce 6\/2020 Doručím Do Archivu\)|Arch"
                r"ív V Sídle Strany|Archvi Strany|Archyv Strany|Arhiv Strany|Ar Hiv Stra"
                r"ny|Bude Zasláno Do Prahy|Mariánské Náměstí 2, Praha 1|Na Morání 360\/3"
                r"|Na Moráni 360\/3, Praha|Na Moráni 360\/3. Praha|Na Moráni 360\/3; Pra"
                r"ha|Na Morání 360\/3, Praha 120 00|Na Moráni 360\/3; Praha 12800|Na Mor"
                r"áni 360\/3, Praha 2|Na Moráni Praha|Na Moráni, Praha|Pirátská Centrum "
                r"Na Moráni 360\/3, Praha|Pirátská Centrum Praha, Na Moráni|Pirátské Cen"
                r"trum, Na Moráni 360\/3, Praha|Pirátské Centrum, Na Moráni 360, Praha|P"
                r"irátské Centrum Praha, Na Moráni|Pirátské Centrum, Řehořova 19|Pirátsk"
                r"é Centrum, Řehořova 19 <!-- Donesl Štěpán Štrébl -->|Pirátské Centrum,"
                r" Řehořova 19, Praha 3|Praha, Kancelář Strany|Řehořova 943\/19 130 00 P"
                r"raha 3|Řehořova 943\/19, 130 00 Praha 3|Řehořova 943\/19, 130 00, Prah"
                r"a 3|Řehořova 943\/19, Praha 3|Sídlo Strany|Sídlo Strany, Řehořova 19, "
                r"Praha 3|Smlouva Uložena Na Místě: Centrální Spisovna|Smlouva Uložena V"
                r" Archivu Strany|Stranický Archív|V Archivu|V Archivukrajského Koordiná"
                r"tora, Později V Archivu Smluv Strany|Zatím Archiv Krajského Koordináto"
                r"ra, Bude Odeslána Do Registru|Zastupitelský Klub Pirátů, Sídlo Strany|"
                r"Kancelář Srtrany|Koks Od 2019 V Archivu|Originál Uložen V Centrálním A"
                r"rchivu Strany|Originál Včetně Osobních Údajů Uložen V Centrálním Archi"
                r"vu Strany|Originál Včetně Podpisů A Osobních Údajů Je Uložen V Centrál"
                r"ním Archivu Strany|Originál Vč. Osobních Údajů A Podpisů Uložen V Cent"
                r"rálním Archivu Strany|Originál Vč. Osobních Údajů Uložen V Centrálním "
                r"Archivu Strany|Pirátské Centrum Praha|Pice Praha|Podepsaný Orgininál V"
                r"č. Osobních Údajů Uložen V Centrálním Archivu Strany)$",
                "Centrální spisovna",
            ),
            (
                r"^(Archiv Pirátské Centrum V Plzni|Archiv Plzen Piratske Centrum|Archiv"
                r" Plzeňské Pirátské Centrum|Archiv V Piratskem Centru Plzen|Archiv V Pl"
                r"zenskem Piratskem Centru|Petr Vileta, Plzeň, Archiv, Sídlo Strany|Petr"
                r" Vileta, Plzeň, Vedoucí Fo|Petr Vileta, Vedoucí Fo, Plzeň|Pice Plzeň|P"
                r"iratske Centrum Plzen|Pirátské Centrum Plzeň|Pirátské Centrum V Plzni|"
                r"Plzeň|Plzenske Piratske Centrum|Plzeňské Pirátské Centrum|V Archivu Pl"
                r"zeňského Kraje|Vedoucí Fo, Plzeň)$",
                "Plzeňská spisovna",
            ),
            (
                r"^(Archiv Klubu|Archiv Pk|Archiv Pk Piráti|Archiv Poslaneckého Klubu|Ar"
                r"chiv Poslaneckého Klubu Piráti|Archiv, Poslanecký Klub Pirátů, Sněmovn"
                r"í 4, 118 00, Praha 1|Kancelář Pirátského Klubu Č. 40, Mhmp, Mariánské "
                r"Náměstí 2, Praha 1|Posklanecký Klub|Poslanecký Kllub|Poslanecký Klub|P"
                r"oslanecký Klub Piráti|Poslanecký Klub Pirátů|Poslanecký Kub|Poslanekcý"
                r" Klub|Poslaneský Klub Piráti)$",
                "Spisovna poslaneckého klubu",
            ),
            (
                r"^(Pirátská Centrum Hradec Králové|Pirátské Centrum Hradec Králové|Koks"
                r" - Khk Kraj - Krapice|Koks - Pirátské Centrum Krapice Hradec Králové|K"
                r"rapice|Krapice Hk|Krapice Hradec Králové|Krapice - Masarykovo Nám. 368"
                r"/1, Hradec Králové|Smlouva Uložena Na Místě: Koks - Khk Kraj - Krapice"
                r")$",
                "Spisovna Hradec Králové",
            ),
            (
                r"^(Archiv Psb|Archiv Psb Brno|Psb Brno|Psb Archiv)$",
                "Brněnská spisovna",
            ),
            (
                r"^(Pice Olomouc|Archiv Olk \(dočasně\)|Pice Olomouc)$",
                "Olomoucká spisovna",
            ),
            (r"^Archiv Ks Vysočina$", "Vysočincká spisovna"),
        )

        for pattern in patterns:
            area_name = re.sub(pattern[0], pattern[1], area_name)

        return area_name

    def normalize_filename(self, filename: str) -> str:
        filename = string.capwords(filename)

        patterns = (
            (r"\s\s+", " "),
            (
                r"^((P|p)odepsaná (V|v)erze|Strojově Čitelná Verze)$",
                "Anonymizovaná verze",
            ),
            (r"^(P|p)odepsaná (V|v)erze 2$", "Anonymizovaná verze 2"),
            (r"^(U|u)pravitelná (V|v)erze$", "Upravitelná verze"),
        )

        for pattern in patterns:
            filename = re.sub(pattern[0], pattern[1], filename)

        return filename

    def parse_index(
        self,
        contract_root: str,
        open_file,
    ) -> dict:
        split_contents = open_file.read().split("---")

        if len(split_contents) < 2:
            raise ValueError(f"{contract_root} index does not have valid metadata.")

        # Use Jan 1, 1970 as the default for invalid years.
        split_contents[1] = (
            split_contents[1]
            .replace("0000-00-00", "1970-01-01")
            .replace("2222-22-22", "1970-01-01")
            .replace("2222-00-00", "1970-01-01")
            .replace("-32", "-31")
            .replace("\t", " ")
        )
        self.normalization_count += 1

        yaml_source = split_contents[1]

        try:
            parsed_metadata = yaml.safe_load(io.StringIO(yaml_source))
        except yaml.YAMLError as exc:
            raise ValueError(
                f"Failed to parse {contract_root} metadata: {exc}."
            ) from exc

        if parsed_metadata is None:
            raise ValueError(f"Got no metadata from {contract_root}.")

        return parsed_metadata

    def assign_signing_party_metadata(
        self, slug: str, contract: Contract, signing_party: dict
    ) -> tuple[
        Contract | Signee,
        list[ContractIssue],
        list[ContracteeSignatureRepresentative | SigneeSignatureRepresentative],
        bool,
        int,
    ]:
        issue_count = 0
        issues = []

        if (
            "jméno" not in signing_party
            and "název" not in signing_party
            and "název společnosti" not in signing_party
        ):
            issue_count += 1
            contract.notes += (
                f"Nepojmenovaná smluvní strana, zdrojová data: {signing_party}\n"
            )
            issues.append(self.use_issue("Špatně pojmenovaná smluvní strana"))

            if self.verbosity >= 2:
                self.stderr.write(
                    self.style.NOTICE(
                        f"Contract {slug} has an unnamed signing party: {signing_party}."
                    )
                )

            return issue_count

        name = signing_party.get(
            "jméno", signing_party.get("název", signing_party.get("název společnosti"))
        ).strip()

        patterns = (
            (r"\s\s+", " "),
            (r"(B|b)c\.|,\s(P|p)h\.(D|d)\.|(M|m)g(r|A|a)\.|(I|i)ng\.|PeadDr\.|PeaDr\.|(P|p)h(D|d)r\.", ""),
            (r"\s(e|E)t\s", ""),
            (r"\s\s+", " "),
            (r"^ ", ""),
            (r"^Alvarium, s\.r\.o\.", "Alvarium s.r.o."),
            (r"^Politické hnutí Senátor 21$", "Politické hnutí SENÁTOR 21"),
            (r"^Petr Springfield$", "Petr Springinsfeld"),
            (r"RAINREKNAM", "RAILREKLAM"),
            (r"^Šárka václavíková$", "Šárka Václavíková"),
            (r"^Štepán Drtina$", "Štěpán Drtina"),
            (r"^(1\. Pirátská s\.r\.o|1\.Pirátská s\.r\.o\.)$", "1. Pirátská s.r.o."),
            (r"České pojišťovna, a.s.", "Česká pojišťovna, a.s."),
            (r"Datrolex, s.r.o.", "DATROLEX, s.r.o."),
            (r"^Jiri ", "Jiří "),
            (
                (
                    r"^(Křesťanská a demokratická unie – Československá strana lidová|"
                    r"Křesťansko demokratická unie – Československá strana lidová|Křes"
                    r"ťanská a demokratická unie - Československá strana lidová)$"
                ),
                "Křesťanská a demokratická unie – Československá strana lidová",
            ),
            (r"^ČSSD$", "Česká strana sociálně demokratická"),
            (r"^Generali česká pojišťovna$", "Generali Česká pojišťovna a.s."),
            (r"LN - Audit s\.r\.o\." "LN-AUDIT s.r.o."),
            (r"Olga Richteová", "Olga Richterová"),
            (
                r"^(politické hnutí Změna|PolitickéHnutí Změna)$",
                "Politické hnutí Změna",
            ),
            (r"^iveta", "Iveta"),
            (r"^Jan Bohm$", "Jan Böhm"),
            (
                r"^Kooperativa$|^Kooperativa pojišťovna, a\.s\.$",
                "Kooperativa pojišťovna, a.s., Vienna Insurance Group"
            ),
            (r"^Dominika P\. Michailidu$", "Dominika Poživilová Michailidu"),
            (r"^Miluš ", "Miluše"),
            (r"^Občanské sdružení o\.s\.$", "Občanské sdružení, o.s."),
            (r"^Aneta Hedilova$", "Aneta Heidlová"),
            (r"^Systemický institut s\.r\.o\$", "Systemický institut, s.r.o."),
            (r"^Václav fořtík$", "Václav Fořtík"),
            (r"^Vodafone$", "Vodafone Czech Republic a.s."),
            (r"^VojtěchHolík$", "Vojtěch Holík"),
            (r"^Vojtech ", "Vojtěch "),
            (r"^Zdenek ", "Zdeněk "),
            (r" Bohmova$", " Bohmová"),
            (r"^Vratislav filípek$", "Vratislav Filípek"),
            (r"^W Czech development$", "W Czech Development s.r.o."),
            (r"^Nadislav", "Ladislav"),
            (r"^Nukáš", "Lukáš"),
            (r"^Jan Nička$", "Jan Lička"),
            (r"^Jan Nipavský$", "Jan Lipavský"),
            (r"^Jan Noužek$", "Jan Loužek"),
            (r"^Simona Nuftová$", "Simona Luftová"),
            (r"^Václav Náska$", "Václav Láska"),
            (r"^Michal Nanger$", "Michal Langer"),
            (r"^Nadislav", "Ladislav"),
            (r"^Naureen Hollge$|^Naureen Holge$|^Naureen Höllge$", "Laureen Höllge"),
            (r"^Oldřich Nhotský$", "Oldřich Lhotský"),
            (r" (KUdláčková|Kudlláčková)$", " Kudláčková"),
            (r"^Jiří knotek$", "Jiří Knotek"),
            (r"^JIří Roubíček$", "Jiří Roubíček"),
            (r"^Koalice Vlasta\. z\.s\.$", "Koalice Vlasta, z.s."),
            (r"^Mikuáš ", "Mikuláš "),
            (r"^Vítězslav Adamec, předseda KS Karlovarský kraj$", "Vítězslav Adamec"),
            (r"^Strana zelených$", "Strana Zelených"),
            (r"^Systemický institut s\.r\.o\.$", "Systemický institut, s.r.o."),
            (r"^Adéla hradilová$|^Adela Hradilova$", "Adéla Hradilová"),
        )

        for pattern in patterns:
            name = re.sub(pattern[0], pattern[1], name)

        self.normalization_count += 1

        is_contractee = False
        representatives = []

        if name.lower() in (
            "česká pirátská strana",
            "česká pirástká strana",
            "česká pirátkská strana",
            "česká pirátská stran",
        ):
            model = Contractee
            representative_model = ContracteeSignatureRepresentative
            instance = model()
            is_contractee = True
        else:
            model = Signee
            representative_model = SigneeSignatureRepresentative
            instance = model(name=name, address_country="Česká republika")

        if model is Signee:
            if (
                "s.r.o" in instance.name
                or "s. r. o." in instance.name
                or "a.s." in instance.name
                or "a. s." in instance.name
                or "o.s." in instance.name
                or "o. s." in instance.name
                or "z.s." in instance.name
                or "z. s." in instance.name
                or "1" in instance.name
                or "2" in instance.name
                or "3" in instance.name
                or "4" in instance.name
                or "5" in instance.name
                or "6" in instance.name
                or "7" in instance.name
                or "8" in instance.name
                or "9" in instance.name
                or "0" in instance.name
                or len(instance.name.split(" ")) != 2
                or instance.name in ("Strana Zelených",)
            ):
                instance.entity_type = instance.EntityTypes.LEGAL_ENTITY
            else:
                if instance.ico_number is None:
                    instance.entity_type = instance.EntityTypes.NATURAL_PERSON
                else:
                    instance.entity_type = instance.EntityTypes.BUSINESS_NATURAL_PERSON

        for signing_party_key, signing_party_value in signing_party.items():
            if isinstance(signing_party_value, str):
                signing_party_value = signing_party_value.strip()

            match signing_party_key:
                case ["sídlo" | "bydliště"]:
                    if is_contractee:
                        continue

                    if not isinstance(signing_party_value, str):
                        issue_count += 1
                        contract.notes += f"Špatně zadané sídlo smluvní strany: {signing_party_value}\n"
                        issues.append(
                            self.use_issue("Špatně zadané sídlo smluvní strany")
                        )

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has an invalid signing party address: {signing_party_value}."
                                )
                            )

                        continue

                    raw_parsed_address_data = parse_address(signing_party_value)
                    address = {}

                    # Convert to a dict first, so that we can access other keys in the loop later
                    for address_info in raw_parsed_address_data:
                        address_value, address_key = address_info
                        address[address_key] = address_value

                    instance.address_street_with_number = ""

                    if "road" in address:
                        instance.address_street_with_number = string.capwords(
                            address["road"]
                        )

                    if "house_number" in address:
                        instance.address_street_with_number += (
                            f" {address['house_number']}"
                        )

                    for address_key, address_value in address.items():
                        match address_key:
                            case "city":
                                if "district" not in address:
                                    instance.address_district = string.capwords(
                                        address_value
                                    )
                            case "house":
                                if "city" not in address and "district" not in address:
                                    instance.address_district = string.capwords(
                                        address_value
                                    )
                            case "city_district":
                                instance.address_district = string.capwords(
                                    address_value
                                )
                            case "postcode":
                                instance.address_zip = string.capwords(address_value)

                    self.normalization_count += 1
                case "IČ":
                    if is_contractee:
                        continue

                    if not isinstance(signing_party_value, int | str):
                        issue_count += 1
                        contract.notes += (
                            f"Špatně zadané IČO smluvní strany: {signing_party_value}\n"
                        )
                        issues.append(
                            self.use_issue("Špatně zadané IČO smluvní strany")
                        )

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has an invalid signing party IČO: {signing_party_value}."
                                )
                            )

                        continue

                    signing_party_value = str(signing_party_value)

                    # Replace up to 5 leading zeroes
                    for i in range(5):
                        if signing_party_value.startswith("0"):
                            self.normalization_count += 1

                            signing_party_value = signing_party_value[1:]

                    instance.ico_number = signing_party_value
                case "zástupce":
                    if not isinstance(signing_party_value, str | list):
                        issue_count += 1
                        contract.notes += f"Špatně zadaný zástupce smluvní strany: {signing_party_value}\n"
                        issues.append(
                            self.use_issue("Špatně zadaný zástupce smluvní strany")
                        )

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has an invalid signing party "
                                    f"representative: {signing_party_value}."
                                )
                            )

                        continue

                    if isinstance(signing_party_value, str):
                        signing_party_value = re.sub(r",$", "", signing_party_value)
                        self.normalization_count += 1

                        function = None

                        if "funkce" in signing_party:
                            if isinstance(signing_party["funkce"], str):
                                function = signing_party["funkce"]
                            else:
                                issue_count += 1
                                contract.notes += f"Špatně zadaná funkce zástupce smluvní strany: {signing_party['funkce']}\n"
                                issues.append(
                                    self.use_issue(
                                        "Špatně zadaná funkce zástupce smluvní strany"
                                    )
                                )

                                if self.verbosity >= 2:
                                    self.stderr.write(
                                        self.style.NOTICE(
                                            f"Contract {slug} has an invalid signing party "
                                            f"representative function: {signing_party['funkce']}."
                                        )
                                    )

                        representatives.append(
                            representative_model(name=signing_party_value)
                        )
                    else:
                        for representative_name in signing_party_value:
                            if not isinstance(representative_name, str):
                                issue_count += 1
                                contract.notes += f"Špatně zadaný jeden ze zástupců smluvní strany: {representative_name}\n"
                                issues.append(
                                    self.use_issue(
                                        "Špatně zadaný zástupce smluvní strany"
                                    )
                                )

                                if self.verbosity >= 2:
                                    self.stderr.write(
                                        self.style.NOTICE(
                                            f"Contract {slug} has an invalid signing party "
                                            f"representative list item: {representative_name}."
                                        )
                                    )

                                continue

                            representative_name = re.sub(r",$", "", representative_name)
                            self.normalization_count += 1

                            representatives.append(
                                representative_model(name=signing_party_value)
                            )
                case "orgán":
                    if not isinstance(signing_party_value, str):
                        issue_count += 1
                        contract.notes += f"Špatně zadaný orgán smluvní strany: {signing_party_value}\n"
                        issues.append(
                            self.use_issue("Špatně zadaný orgán smluvní strany")
                        )

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has an invalid signing party "
                                    f"department: {signing_party_value}."
                                )
                            )

                        continue

                    if signing_party_value in (
                        "Česká Pirátská strana",
                        "Česká pirátská strana",
                        "Čeksá pirátská strana",
                        "piráti",
                        "Piráti"
                    ) or signing_party_value == instance.name:
                        # Irrelevant
                        self.normalization_count += 1
                        continue

                    if signing_party_value and model is Signee and instance.entity_type == instance.EntityTypes.NATURAL_PERSON:
                        issue_count += 1
                        contract.notes += f"Fyzická osoba má zadaný orgán: {signing_party_value}\n"
                        issues.append(
                            self.use_issue("Chybná organizační složka")
                        )

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} signed by a natural person "
                                    f"has a department: {signing_party_value}."
                                )
                            )

                        continue

                    if signing_party_value in (
                        "dle stanov a na základě plné moci",
                    ):
                        issue_count += 1
                        contract.notes += f"Chybná organizační složka: {signing_party_value}\n"
                        issues.append(
                            self.use_issue("Chybná organizační složka")
                        )

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has an invalid department: "
                                    f"{signing_party_value}."
                                )
                            )

                        continue

                    instance.department = self.normalize_department(signing_party_value)
                    self.normalization_count += 1

        if model is Signee:
            if (
                "s.r.o" in instance.name
                or "s. r. o." in instance.name
                or "a.s." in instance.name
                or "a. s." in instance.name
                or "o.s." in instance.name
                or "o. s." in instance.name
                or "z.s." in instance.name
                or "z. s." in instance.name
                or "1" in instance.name
                or "2" in instance.name
                or "3" in instance.name
                or "4" in instance.name
                or "5" in instance.name
                or "6" in instance.name
                or "7" in instance.name
                or "8" in instance.name
                or "9" in instance.name
                or "0" in instance.name
                or len(instance.name.split(" ")) != 2
                or instance.name in ("Strana Zelených",)
            ):
                instance.entity_type = instance.EntityTypes.LEGAL_ENTITY
            else:
                if instance.ico_number is None:
                    instance.entity_type = instance.EntityTypes.NATURAL_PERSON
                else:
                    instance.entity_type = instance.EntityTypes.BUSINESS_NATURAL_PERSON

        # Do our best to merge signing parties together.
        existing_instances = model.objects.filter(
            (
                models.Q(name=instance.name)
                & (
                    models.Q(department=instance.department)
                    if instance.department is not None
                    else models.Q(department__isnull=True)
                )
                & (
                    (
                        models.Q(
                            address_street_with_number=instance.address_street_with_number
                        )
                        if instance.address_street_with_number is not None
                        else models.Q(address_street_with_number__isnull=True)
                    )
                    | (
                        models.Q(date_of_birth=instance.date_of_birth)
                        if model is Signee and instance.date_of_birth is not None
                        else (
                            models.Q(date_of_birth__isnull=True)
                            if model is Signee
                            else models.Value(False)
                        )
                    )
                )
            )
            | (
                (
                    models.Q(ico_number=instance.ico_number)
                    & (
                        models.Q(department=instance.department)
                        if instance.department is not None
                        else models.Q(department__isnull=True)
                    )
                )
                if instance.ico_number is not None
                else models.Value(False)
            )
        ).all()

        if len(existing_instances) != 0:
            for position, existing_instance in enumerate(existing_instances):
                if (
                    existing_instance.ico_number is None
                    and instance.ico_number is not None
                ):
                    existing_instance.ico_number = instance.ico_number
                    existing_instance.save()
                    instance = existing_instance
                    break
                elif (
                    existing_instance.ico_number == instance.ico_number
                    or instance.ico_number is None
                ):
                    instance = existing_instance
                    break
                elif position == len(existing_instances) - 1:
                    instance.save()
        else:
            instance.save()

        return instance, issues, representatives, is_contractee, issue_count

    def assign_contract_data(
        self,
        contract: Contract,
        metadata: dict,
        slug: str,
        contract_root: str,
    ) -> None:
        filing_area = None
        types = []
        signees = []
        contractees = []
        files = []
        issues = []
        is_already_imported = False
        observed_issues_count = 0

        for key, value in metadata.items():
            key = key.strip()

            if isinstance(value, str):
                value = value.strip()

            match key:
                case "datum účinnosti":
                    if isinstance(value, date):
                        if value.year not in (1970, 2100):  # Ignore placeholder years
                            contract.valid_start_date = value
                    elif value is not None:
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadaný začátek platnosti: {value}\n"
                        issues.append(self.use_issue("Špatně zadaný začátek platnosti"))

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has a broken valid start date: {value}."
                                )
                            )
                case "datum ukončení":
                    if isinstance(value, date):
                        if value.year not in (1970, 2100):  # Ignore placeholder years
                            contract.valid_end_date = value
                    elif value is not None and (
                        isinstance(value, str) and value.lower() != "na dobu neurčitou"
                    ):
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadaný konec platnosti: {value}\n"
                        issues.append(self.use_issue("Špatně zadaný konec platnosti"))

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has a broken valid end date: {value}."
                                )
                            )
                case "title":
                    contract.name = value

                    if Contract.objects.filter(name=value).exists():
                        value += f"(DUPLIKÁT - {slug})"

                    if Contract.objects.filter(name=value).exists():
                        is_already_imported = True
                        if self.verbosity >= 1:
                            self.stdout.write(f"{slug} already exists, skipping.")

                        break
                case "použité smluvní typy":
                    if isinstance(value, str):
                        value = self.normalize_type(value)
                        self.normalization_count += 1

                        try:
                            type_instance = ContractType.objects.get(name=value)
                        except ContractType.DoesNotExist:
                            type_instance = ContractType(name=value)

                        types.append(type_instance)

                        continue
                    elif not isinstance(value, list):
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadané typy: {value}\n"
                        issues.append(self.use_issue("Špatně zadané typy"))

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} is missing types - not a list: {value}."
                                )
                            )

                        continue

                    for type_name in value:
                        if not isinstance(type_name, str):
                            observed_issues_count += 1
                            contract.notes += f"Špatně zadaný typ: {type_name}\n"
                            issues.append(self.use_issue("Špatně zadaný typ"))

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} is missing types - list item is not a string: {value}."
                                    )
                                )

                            continue

                        type_name = self.normalize_type(type_name)
                        self.normalization_count += 1

                        try:
                            type_instance = ContractType.objects.get(name=type_name)
                        except ContractType.DoesNotExist:
                            type_instance = ContractType(name=type_name)

                        types.append(type_instance)
                case "předmět":
                    contract.summary = value
                case "stav":
                    lower_value = value.lower()

                    if (
                        lower_value.startswith("splněn")
                        or lower_value.startswith("ukončen")
                        or lower_value.startswith("vypovězen")
                        or lower_value.startswith("odstoupen")
                        or lower_value in ("spněno",)
                    ):
                        contract.is_valid = False
                    elif (
                        lower_value.startswith("platn")
                        or lower_value.startswith("řešen")
                        or lower_value in ("v plnění", "v řešení")
                    ):
                        contract.is_valid = True
                    else:
                        observed_issues_count += 1
                        contract.notes += f"Neznámý stav: {value}\n"
                        issues.append(self.use_issue("Neznámý právní stav"))

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has an invalid state: {value}."
                                )
                            )
                case "náklady":
                    if isinstance(value, str):
                        formatted_value = value.replace(" ", "").replace(".00", ".0")

                        if formatted_value.isnumeric():
                            value = float(formatted_value)

                    if isinstance(value, (int, float)):
                        if value < 0:
                            observed_issues_count += 1
                            contract.notes += (
                                f"Původní, špatně zadané náklady: {value}\n"
                            )
                            issues.append(self.use_issue("Špatně zadané náklady"))

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} has an invalid cost amount: {value}."
                                    )
                                )

                            continue

                        if value != 0:
                            contract.cost_amount = int(value)
                            contract.cost_unit = contract.CostUnits.TOTAL
                    elif isinstance(value, str):
                        value = value.lower().replace(",-", "")

                        if not value.endswith("kč/h"):
                            continue

                        split_value = value.split("kč/h")

                        if (
                            len(split_value) != 2
                            or not split_value[0].strip().isnumeric()
                        ):
                            observed_issues_count += 1
                            contract.notes += f"Původní, neropoznané náklady: {value}\n"
                            issues.append(self.use_issue("Špatně zadané náklady"))

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Could not parse cost for contract {slug}: {value}."
                                    )
                                )

                            continue

                        contract.cost_amount = int(split_value[0].strip())
                        contract.cost_unit = contract.CostUnits.HOUR
                    elif value not in (None, "0"):
                        observed_issues_count += 1
                        contract.notes += f"Původní, neropoznané náklady: {value}\n"
                        issues.append(self.use_issue("Špatně zadané náklady"))

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Could not parse cost for contract {slug}: {value}."
                                )
                            )
                case "místo uložení":
                    if isinstance(value, str):
                        value = self.normalize_filing_area(value)
                        self.normalization_count += 1

                    try:
                        contract.paper_form_state = contract.PaperFormStates.STORED
                        filing_area = ContractFilingArea.objects.get(name=value)
                    except ContractFilingArea.DoesNotExist:
                        if isinstance(value, str) and not value.startswith("Neznám"):
                            filing_area = ContractFilingArea(
                                name=value,
                                person_responsible="Doplň osobu!",
                            )
                        else:
                            observed_issues_count += 1
                            contract.notes += f"Špatně zadaná spisovna: {value}\n"
                            issues.append(self.use_issue("Špatně zadaná spisovna"))

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} has an invalid filing area: {value}."
                                    )
                                )
                case "smluvní strany":
                    if not isinstance(value, list):
                        observed_issues_count += 1
                        contract.notes += (
                            f"Špatně zadané smluvní strany, nejsou seznam: {value}\n"
                        )
                        issues.append(self.use_issue("Špatně zadaný smluvní strany"))

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Signing parties on {slug} are not a list: {value}."
                                )
                            )

                        continue

                    for signing_party in value:
                        if not isinstance(signing_party, dict):
                            observed_issues_count += 1
                            contract.notes += (
                                f"Špatně zadaná smluvní strana: {signing_party}\n"
                            )
                            issues.append(
                                self.use_issue("Špatně zadaná smluvní strana")
                            )

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Signing party on {slug} is not a dictionary: {value}."
                                    )
                                )

                            continue

                        (
                            instance,
                            signing_party_issues,
                            representatives,
                            is_contractee,
                            signing_party_issue_count,
                        ) = self.assign_signing_party_metadata(
                            slug, contract, signing_party
                        )

                        issues += signing_party_issues
                        observed_issues_count += signing_party_issue_count

                        # Store representatives in relation to the instance, hacky but good enough
                        instance._representatives = representatives

                        if is_contractee:
                            contractees.append(instance)
                        else:
                            signees.append(instance)
                case "soubory":
                    if not isinstance(value, list):
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadané soubory.\n"
                        issues.append(self.use_issue("Špatně zadané soubory"))

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Files for {slug} are not a list: {value}."
                                )
                            )

                        continue

                    at_least_one_incorrect_filename = False

                    for file_data in value:
                        if not isinstance(file_data, dict):
                            observed_issues_count += 1
                            contract.notes += (
                                f"Špatně zadané informace o souboru: {file_data}.\n"
                            )
                            issues.append(
                                self.use_issue("Špatně zadané informace o souboru")
                            )

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"File data in {slug} is not a dict: {file_data}."
                                    )
                                )

                            continue

                        for file_key, file_value in file_data.items():
                            file_key = file_key.strip()

                            if file_key.lower() in ("název", "náhled", "náhlad"):
                                continue

                            if not isinstance(file_value, str):
                                observed_issues_count += 1
                                contract.notes += f"Špatně zadaný název souboru {file_key}: {file_value}.\n"

                                if not at_least_one_incorrect_filename:
                                    issues.append(
                                        self.use_issue("Neplatný název souboru")
                                    )

                                if self.verbosity >= 2:
                                    self.stderr.write(
                                        self.style.NOTICE(
                                            f"Filename in {slug} for file {file_key} invalid: {file_value}."
                                        )
                                    )

                                at_least_one_incorrect_filename = True

                                continue

                            file_path = os.path.join(contract_root, file_value)

                            if not os.path.isfile(file_path):
                                observed_issues_count += 1
                                contract.notes += (
                                    f"Neexistující soubor: {file_value}.\n"
                                )
                                issues.append(self.use_issue("Neexistující soubor"))

                                if self.verbosity >= 2:
                                    self.stderr.write(
                                        self.style.NOTICE(
                                            f"Filename in {slug} does not correspond to a file: {file_value}."
                                        )
                                    )

                                continue

                            with open(file_path, "rb") as open_file:
                                self.normalization_count += 1
                                file = ContractFile(
                                    contract=contract,
                                    name=self.normalize_filename(file_key),
                                    is_public=True,
                                )

                                file.file.save(
                                    file_value,
                                    File(open_file),
                                    save=False,
                                )

                                files.append(file)

        if not is_already_imported:
            if contract.name in (None, "") or (
                isinstance(contract.name, str)
                and re.sub(r"/\s\s+/", "", contract.name) == ""
            ):
                self.normalization_count += 1
                contract.name = slug

            if contract.valid_start_date is None:
                contract.valid_start_date = date(year=1970, month=1, day=1)

            if observed_issues_count != 0:
                self.partial_import_count += 1
                self.issue_count += observed_issues_count
            else:
                self.normal_import_count += 1

            if filing_area is not None:
                filing_area.save()

            for type_ in types:
                type_.save()

            # Save primary key first
            contract.save()

            for contractee in contractees:
                contractee.save()

                signature = ContracteeSignature(
                    contract=contract,
                    contractee=contractee,
                    date=contract.valid_start_date,
                )

                signature.save()

                contract.contractee_signatures.add(signature)

                for representative in contractee._representatives:
                    representative.signature = signature
                    representative.save()

            for signee in signees:
                signee.save()

                signature = SigneeSignature(
                    contract=contract,
                    signee=signee,
                    date=contract.valid_start_date,
                )

                signature.save()

                contract.signee_signatures.add(signature)

                for representative in signee._representatives:
                    representative.signature = signature
                    representative.save()

            for file in files:
                file.save()

            contract.filing_area = filing_area
            contract.types.set(types)
            contract.issues.set(issues)
            contract.save()
        else:
            self.already_imported_count += 1

    def import_contract_from_files(
        self, contract_root: str, files: list[str], valid_start_date: datetime
    ) -> None:
        contract = Contract(notes="")

        for file_ in files:
            with open(
                os.path.join(
                    contract_root,
                    file_,
                ),
                "r",
            ) as open_file:
                if file_ == "index.html":
                    metadata_failed = True

                    try:
                        metadata = self.parse_index(contract_root, open_file)
                    except ValueError as exc:
                        if self.verbosity >= 1:
                            self.stderr.write(
                                self.style.WARNING(
                                    f"Could not parse {contract_root} metadata: {exc}"
                                )
                            )

                        self.fatal_error_count += 1

                        return

                    self.assign_contract_data(
                        contract,
                        metadata,
                        os.path.basename(contract_root),
                        contract_root,
                    )
                elif file_.endswith(".pdf"):
                    # TODO
                    continue

    def import_all_contracts(self, git_dir) -> None:
        year_root = os.path.join(
            git_dir,
            "smlouvy",
        )

        for year_directory in sorted(os.listdir(year_root)):
            if int(year_directory) == 0:
                continue  # Out of range, TODO

            month_root = os.path.join(
                year_root,
                year_directory,
            )

            for month_directory in sorted(os.listdir(month_root)):
                day_root = os.path.join(
                    month_root,
                    month_directory,
                )

                for day_directory in sorted(os.listdir(day_root)):
                    contract_root = os.path.join(
                        git_dir,
                        "smlouvy",
                        year_directory,
                        month_directory,
                        day_directory,
                    )

                    for contract_directory in sorted(os.listdir(contract_root)):
                        this_contract_directory = os.path.join(
                            contract_root,
                            contract_directory,
                        )

                        if not os.path.isdir(this_contract_directory):
                            if self.verbosity >= 1:
                                self.stderr.write(
                                    self.style.WARNING(
                                        f"{this_contract_directory} is not a directory and thus invalid, skipping."
                                    )
                                )

                            self.fatal_error_count += 1

                            continue

                        valid_start_date = datetime(
                            year=int(year_directory),
                            month=int(month_directory),
                            day=int(day_directory),
                        )

                        self.import_contract_from_files(
                            this_contract_directory,
                            os.listdir(this_contract_directory),
                            valid_start_date,
                        )

        if self.verbosity >= 1:
            self.stdout.write(
                self.style.SUCCESS(
                    "\n"
                    f"Saved a total of {self.normal_import_count + self.partial_import_count} contracts.\n"
                    f"    {self.partial_import_count} contained a total of {self.issue_count} issues.\n"
                    f"    {self.already_imported_count} were already saved previously and skipped.\n"
                    f"    {self.fatal_error_count} potential contracts were unparseable.\n"
                    f"    {self.normalization_count} data points were normalized."
                )
            )

    def handle(self, *args, **options) -> None:
        self.verbosity = options["verbosity"]

        git_dir = os.path.join(
            os.getcwd(),
            (options["directory"] if options["directory"] is not None else "git"),
        )

        if os.path.exists(git_dir) and not options["delete"]:
            if not options["existing"]:
                if self.verbosity >= 1:
                    self.stderr.write(
                        self.style.ERROR(
                            f"Temporary git storage directory ({git_dir}) already exists.\n"
                            "As it could contain other data, it will not be removed.\n"
                            "Please remove it manually and try again or use the '--existing' "
                            "argument."
                        )
                    )

                return
            else:
                if self.verbosity >= 2:
                    self.stdout.write("Using existing git storage directory.")
        else:
            if options["delete"] and os.path.exists(git_dir):
                if self.verbosity >= 2:
                    self.stdout.write("Deleting old git storage directory.")

                shutil.rmtree(git_dir)

            if self.verbosity >= 2:
                self.stdout.write("Cloning repository.")

            Repo.clone_from(
                options["repo_url"],
                git_dir,
                branch=options["branch"],
            )

            if self.verbosity >= 1:
                self.stdout.write(self.style.SUCCESS("Finished cloning repository."))

        if options["purge"]:
            for model in (
                Contract,
                ContractType,
                ContractFilingArea,
                ContractIssue,
                ContractFile,
                Contractee,
                ContracteeSignature,
                Signee,
                SigneeSignature,
            ):
                model.objects.filter().delete()

            if self.verbosity >= 1:
                self.stdout.write(self.style.SUCCESS("Deleted all previous records."))

        if self.verbosity >= 2:
            self.stdout.write("\n")
        self.import_all_contracts(git_dir)

        if self.verbosity >= 1:
            self.stdout.write(self.style.SUCCESS("\nGit repository sync complete."))
