import io
import os
import re
import string
from datetime import date, datetime

import yaml
from django.conf import settings
from django.core.management.base import BaseCommand

from git import Repo

from ...models import Contract, ContractFilingArea, ContractType


class Command(BaseCommand):
    help = "Sync contract data from a remote Git repository."

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.normal_import_count = 0
        self.partial_import_count = 0
        self.already_imported_count = 0
        self.issue_count = 0
        self.fatal_error_count = 0

    def add_arguments(self, parser):
        parser.add_argument(
            "repo_url",
            type=str,
            help="URL of the Git repository to clone",
        )
        parser.add_argument(
            "branch",
            type=str,
            help="Branch to use",
        )
        parser.add_argument(
            "--directory",
            type=str,
            help="Directory to store the cloned repository in",
        )
        parser.add_argument(
            "--existing",
            action="store_true",
            help="Use the existing storage directory, as long as it exists.",
        )
        parser.add_argument(
            "--purge",
            action="store_true",
            help="Purge all previous contracts, types and filing areas before the import.",
        )

    def normalize_type(self, type_name: str) -> str:
        type_name = string.capwords(type_name)

        patterns = (
            (r" O ", " o "),
            (r" S ", " s "),
            (r" K ", " k "),
            (r" V ", " v "),
            (r"Nda", "NDA"),
            (r"Dds", "DDS"),
            (r"^Dohoda o Podpoře.*$", "Dohoda o Podpoře"),
            (
                r"^(Dohoda o Ochraně Důvěrných Informací \(nda\)|Dohoda o Ochraně Důvě"
                r"rných Dat|Dohoda o Ochraně Důvěrných Informací|Smlouva o Zachování D"
                r"ůvěrnosti Informací|Smlouva o Zachování Důvěrnosti Informací \(nda\)"
                ")$",
                "NDA",
            ),
            (r"^(Darovací.*|Ds)$", "Darovací Smlouva"),
            (r".*Dodatek.*", "Dodatek"),
            (
                r"^(Dpp|Dohoda o Provední Práce|Dohoda o Proveení Práce)$",
                "Dohoda o Provedení Práce",
            ),
            (r"^Dpč$", "Dohoda o Pracovní Činnosti"),
            (r"^NS$", "Nájemní Smlouva"),
            (r"^Koaliční Smlouva.*$", "Koaliční Smlouva"),
            (r"^Pronajem$", "Pronájem"),
            (
                r"^(Rácová Smlouva|Rámcová Kupní Smlouva|Rámcová Smlouva|Rámcová Smlouva- Do"
                r"datek|Ramcova Smlouva o Najmu Kopií Filmů A Rámcová Smlouva Podlicenční|Rs"
                r")$",
                "Rámcová Smlouva",
            ),
            (
                r"^(Oof|Oosf|Osf|Potvrzení o Funkci|Potvrzeni o Stranicke Funkci|Potvrzení o"
                r"Stranické Funkci)",
                "Osvědčení o Stranické Funkci",
            ),
            (r"^Smlouva o Bezúročné Zápůjčce.*$", "Bezúročná Zápůjčka Peněz"),
            (
                r"^(Smlouva o Dilo|Smlouva o Dílo A Veřejná Licenční Smlouva|Smlouva o Dílo - "
                r"Dodatek|Sod)$",
                "Smlouva o Dílo",
            ),
            (r"^Smlouva o Nájmu.*$", "Smlouva o Nájmu"),
            (
                r"^Smlouva o Poskytovani Pravnich Sluzeb$",
                "Smlouva o Poskytování Právních Služeb",
            ),
            ("\.$", ""),
            (r"^Ks$", "Koaliční Smlouva"),
            (r"^Pm$", "Plná moc"),
            (r"Pronajmu", "Pronájmu"),
            (r"^(Zl .*|(Zadavací|Zdávací) list)$", "Zadávací list"),
            (r"Hpp", "Hlavní Pracovní Poměr"),
            (r"^Sops - Smlouva o Poskytnutí Služeb", "Smlouva o Poskytnutí Služeb"),
            (r"^Suv$", "Smlouva o Uměleckém Vystoupení"),
            (
                r"^(Příkaz|Příkazní|Prikazni Smlouva|Příkazní Smlouva|Příkazní Smlouva, Bude"
                r"Fakturováno|Příkazní Smlouva (dodatek)|Příkazní Smlouva - Dodatek|Příkazní"
                r"Smlouva Dodatek|Příkazní Smlouva- Dodatek|Příkazní Smlouva - Dodatek Č\. 1"
                r"|Příkazní Smlouva - Dodatek Č\.1|Příkazní Smlouva Dodatek Č\. 1|Příkazní S"
                r"mlouva Dodatek Č\.1|Příkazní Smlouva s Dodavatelem|Příkazní Smlouva- Ukonč"
                r"ení|Příkazní Smlouvy|Příkazní Smlova|Příkazní Smluva|Příkazní Smuva|Příkaz"
                r"ní Spisovna|Příkzaní Smlouva|Ps)$",
                "Příkazní smlouva",
            ),
        )

        for pattern in patterns:
            type_name = re.sub(pattern[0], pattern[1], type_name)

        return type_name

    def normalize_filing_area(self, area_name: str) -> str:
        area_name = string.capwords(area_name)

        patterns = (
            (
                r"^(Cenrální Spisovna|Censtrální Spisovna|Centrála|Centrálách Archiv Str"
                r"any|Centrála Strany|Centrální Achiv Strany|Centrální Archiv|Centralni "
                r"Archiv Strany|Centrální Archiv Strany|Centrální Skupina|Centrální Spid"
                r"ovna|Centrální Spisivna|Centrální Spisovna|Centrální Spísovna|Centráln"
                r"í Spisovna Strany|Centrální Sposovna|Centrální Sposovná|Centrální Úlož"
                r"na|Centrální Úschovna|Česká Pirítská Strana, Na Moráni 360\/3, Praha 2"
                r"|Cnterální Spisovna|Archiv|Archív|Archiv Centrály|Archiv České Pirátsk"
                r"é Strany|Archiv Pice|Archív Piráti|Archiv Pirátské Strany|Archiv Praha"
                r"|Archiv, Sídlo Strany|Archiv, Sídlo Strany,|Archív, Sídlo Strany|Archi"
                r"v, Sídlo Strany, Řehořova 19, Praha 3|Archiv, Sídlo Strany, Řehořova 1"
                r"9, Praha 3, Zastupitelský Klub Pirátů, Mariánské Nám. 2, Praha 1 \/pří"
                r"lohy|Archiv, Sídlo Strany, Řehořova 943\/19, Praha 3|Archív, Sídlo Str"
                r"any, Řehořova 943\/19, Praha 3|Archiv, Sídlo Strany, Řehova 19, Praha "
                r"3|Archiv Služeb|Archiv Smlouvy|Archiv Starny|Archiv Strany|Archív Stra"
                r"ny|Archiv Strany, Řehořova 943\/19, Praha|Archiv Strany, Sídlo Strany|"
                r"Archiv Strany \(zatím Koks, Do Konce 6\/2020 Doručím Do Archivu\)|Arch"
                r"ív V Sídle Strany|Archvi Strany|Archyv Strany|Arhiv Strany|Ar Hiv Stra"
                r"ny|Bude Zasláno Do Prahy|Mariánské Náměstí 2, Praha 1|Na Morání 360\/3"
                r"|Na Moráni 360\/3, Praha|Na Moráni 360\/3. Praha|Na Moráni 360\/3; Pra"
                r"ha|Na Morání 360\/3, Praha 120 00|Na Moráni 360\/3; Praha 12800|Na Mor"
                r"áni 360\/3, Praha 2|Na Moráni Praha|Na Moráni, Praha|Pirátská Centrum "
                r"Na Moráni 360\/3, Praha|Pirátská Centrum Praha, Na Moráni|Pirátské Cen"
                r"trum, Na Moráni 360\/3, Praha|Pirátské Centrum, Na Moráni 360, Praha|P"
                r"irátské Centrum Praha, Na Moráni|Pirátské Centrum, Řehořova 19|Pirátsk"
                r"é Centrum, Řehořova 19 <!-- Donesl Štěpán Štrébl -->|Pirátské Centrum,"
                r" Řehořova 19, Praha 3|Praha, Kancelář Strany|Řehořova 943\/19 130 00 P"
                r"raha 3|Řehořova 943\/19, 130 00 Praha 3|Řehořova 943\/19, 130 00, Prah"
                r"a 3|Řehořova 943\/19, Praha 3|Sídlo Strany|Sídlo Strany, Řehořova 19, "
                r"Praha 3|Smlouva Uložena Na Místě: Centrální Spisovna|Smlouva Uložena V"
                r" Archivu Strany|Stranický Archív|V Archivu|V Archivukrajského Koordiná"
                r"tora, Později V Archivu Smluv Strany|Zatím Archiv Krajského Koordináto"
                r"ra, Bude Odeslána Do Registru|Zastupitelský Klub Pirátů, Sídlo Strany|"
                r"Kancelář Srtrany|Koks Od 2019 V Archivu|Originál Uložen V Centrálním A"
                r"rchivu Strany|Originál Včetně Osobních Údajů Uložen V Centrálním Archi"
                r"vu Strany|Originál Včetně Podpisů A Osobních Údajů Je Uložen V Centrál"
                r"ním Archivu Strany|Originál Vč. Osobních Údajů A Podpisů Uložen V Cent"
                r"rálním Archivu Strany|Originál Vč. Osobních Údajů Uložen V Centrálním "
                r"Archivu Strany|Pirátské Centrum Praha|Pice Praha|Podepsaný Orgininál V"
                r"č. Osobních Údajů Uložen V Centrálním Archivu Strany)$",
                "Centrální spisovna",
            ),
            (
                r"^(Archiv Pirátské Centrum V Plzni|Archiv Plzen Piratske Centrum|Archiv"
                r" Plzeňské Pirátské Centrum|Archiv V Piratskem Centru Plzen|Archiv V Pl"
                r"zenskem Piratskem Centru|Petr Vileta, Plzeň, Archiv, Sídlo Strany|Petr"
                r" Vileta, Plzeň, Vedoucí Fo|Petr Vileta, Vedoucí Fo, Plzeň|Pice Plzeň|P"
                r"iratske Centrum Plzen|Pirátské Centrum Plzeň|Pirátské Centrum V Plzni|"
                r"Plzeň|Plzenske Piratske Centrum|Plzeňské Pirátské Centrum|V Archivu Pl"
                r"zeňského Kraje|Vedoucí Fo, Plzeň)$",
                "Plzeňská spisovna",
            ),
            (
                r"^(Archiv Klubu|Archiv Pk|Archiv Pk Piráti|Archiv Poslaneckého Klubu|Ar"
                r"chiv Poslaneckého Klubu Piráti|Archiv, Poslanecký Klub Pirátů, Sněmovn"
                r"í 4, 118 00, Praha 1|Kancelář Pirátského Klubu Č. 40, Mhmp, Mariánské "
                r"Náměstí 2, Praha 1|Posklanecký Klub|Poslanecký Kllub|Poslanecký Klub|P"
                r"oslanecký Klub Piráti|Poslanecký Klub Pirátů|Poslanecký Kub|Poslanekcý"
                r" Klub|Poslaneský Klub Piráti)$",
                "Spisovna poslaneckého klubu",
            ),
            (
                r"^(Pirátská Centrum Hradec Králové|Pirátské Centrum Hradec Králové|Koks"
                r" - Khk Kraj - Krapice|Koks - Pirátské Centrum Krapice Hradec Králové|K"
                r"rapice|Krapice Hk|Krapice Hradec Králové|Krapice - Masarykovo Nám. 368"
                r"/1, Hradec Králové|Smlouva Uložena Na Místě: Koks - Khk Kraj - Krapice"
                r")$",
                "Spisovna Hradec Králové",
            ),
            (
                r"^(Archiv Psb|Archiv Psb Brno|Psb Brno|Psb Archiv)$",
                "Brněnská spisovna",
            ),
            (
                r"^(Pice Olomouc|Archiv Olk \(dočasně\)|Pice Olomouc)$",
                "Olomoucká spisovna",
            ),
            (r"^Archiv Ks Vysočina$", "Vysočincká spisovna"),
        )

        for pattern in patterns:
            area_name = re.sub(pattern[0], pattern[1], area_name)

        return area_name

    def parse_index(
        self,
        contract_root: str,
        open_file,
    ) -> dict:
        split_contents = open_file.read().split("---")

        if len(split_contents) < 2:
            raise ValueError(f"{contract_root} index does not have valid metadata.")

        # Use Jan 1, 1970 as the default for invalid years.
        split_contents[1] = (
            split_contents[1]
            .replace("0000-00-00", "1970-01-01")
            .replace("2222-22-22", "1970-01-01")
            .replace("2222-00-00", "1970-01-01")
            .replace("-32", "-31")
            .replace("\t", " ")
        )

        yaml_source = split_contents[1]

        try:
            parsed_metadata = yaml.safe_load(io.StringIO(yaml_source))
        except yaml.YAMLError as exc:
            raise ValueError(
                f"Failed to parse {contract_root} metadata: {exc}."
            ) from exc

        if parsed_metadata is None:
            raise ValueError(f"Got no metadata from {contract_root}.")

        return parsed_metadata

    def assign_contract_metadata(
        self,
        contract: Contract,
        metadata: dict,
        slug: str,
    ) -> None:
        filing_area = None
        types = []
        is_already_imported = False
        observed_issues_count = 0

        for key, value in metadata.items():
            key = key.strip()

            if isinstance(value, str):
                value = value.strip()

            match key:
                case "datum účinnosti":
                    if isinstance(value, date):
                        if value.year not in (1970, 2100):  # Ignore placeholder years
                            contract.valid_start_date = value
                    elif value is not None:
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadaný začátek platnosti: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has a broken valid start date: {value}."
                                )
                            )
                case "datum ukončení":
                    if isinstance(value, date):
                        if value.year not in (1970, 2100):  # Ignore placeholder years
                            contract.valid_end_date = value
                    elif value is not None and (
                        isinstance(value, str) and value.lower() != "na dobu neurčitou"
                    ):
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadaný konec platnosti: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has a broken valid end date: {value}."
                                )
                            )
                case "title":
                    contract.name = value

                    if Contract.objects.filter(name=value).exists():
                        value += f"(DUPLIKÁT - {slug})"

                    if Contract.objects.filter(name=value).exists():
                        is_already_imported = True
                        if self.verbosity >= 1:
                            self.stdout.write(f"{slug} already exists, skipping.")

                        break
                case "použité smluvní typy":
                    if isinstance(value, str):
                        value = self.normalize_type(value)

                        try:
                            type_instance = ContractType.objects.get(name=value)
                        except ContractType.DoesNotExist:
                            type_instance = ContractType(name=value)

                        types.append(type_instance)

                        continue
                    elif not isinstance(value, list):
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadané typy: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} is missing types - not a list: {value}."
                                )
                            )

                        continue

                    for type_name in value:
                        if not isinstance(type_name, str):
                            observed_issues_count += 1
                            contract.notes += f"Nezaevidovaný typ: {type_name}\n"

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} is missing types - list item is not a string: {value}."
                                    )
                                )

                            continue

                        type_name = self.normalize_type(type_name)

                        try:
                            type_instance = ContractType.objects.get(name=type_name)
                        except ContractType.DoesNotExist:
                            type_instance = ContractType(name=type_name)

                        types.append(type_instance)
                case "předmět":
                    contract.summary = value
                case "stav":
                    lower_value = value.lower()

                    if (
                        lower_value.startswith("splněn")
                        or lower_value.startswith("ukončen")
                        or lower_value.startswith("vypovězen")
                        or lower_value.startswith("odstoupen")
                        or lower_value in ("spněno",)
                    ):
                        contract.is_valid = False
                    elif (
                        lower_value.startswith("platn")
                        or lower_value.startswith("řešen")
                        or lower_value in ("v plnění", "v řešení")
                    ):
                        contract.is_valid = True
                    else:
                        observed_issues_count += 1
                        contract.notes += f"Neznámý stav: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has an invalid state: {value}."
                                )
                            )
                case "náklady":
                    if isinstance(value, str):
                        formatted_value = value.replace(" ", "").replace(".00", ".0")

                        if formatted_value.isnumeric():
                            value = float(formatted_value)

                    if isinstance(value, (int, float)):
                        if value < 0:
                            observed_issues_count += 1
                            contract.notes += (
                                f"Původní, špatně zadané náklady: {value}\n"
                            )

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} has an invalid cost amount: {value}."
                                    )
                                )

                            continue

                        if value != 0:
                            contract.cost_amount = int(value)
                            contract.cost_unit = contract.CostUnits.TOTAL
                    elif isinstance(value, str):
                        value = value.lower().replace(",-", "")

                        if not value.endswith("kč/h"):
                            continue

                        split_value = value.split("kč/h")

                        if (
                            len(split_value) != 2
                            or not split_value[0].strip().isnumeric()
                        ):
                            observed_issues_count += 1
                            contract.notes += f"Původní, neropoznané náklady: {value}\n"

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Could not parse cost for contract {slug}: {value}."
                                    )
                                )

                            continue

                        contract.cost_amount = int(split_value[0].strip())
                        contract.cost_unit = contract.CostUnits.HOUR
                    elif value not in (None, "0"):
                        observed_issues_count += 1
                        contract.notes += f"Původní, neropoznané náklady: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Could not parse cost for contract {slug}: {value}."
                                )
                            )
                case "místo uložení":
                    if isinstance(value, str):
                        value = self.normalize_filing_area(value)

                    try:
                        contract.paper_form_state = contract.PaperFormStates.STORED
                        filing_area = ContractFilingArea.objects.get(name=value)
                    except ContractFilingArea.DoesNotExist:
                        if isinstance(value, str) and not value.startswith("Neznám"):
                            filing_area = ContractFilingArea(
                                name=value,
                                person_responsible="Doplň osobu!",
                            )
                        else:
                            observed_issues_count += 1
                            contract.notes += f"Špatně zadaná spisovna: {value}\n"

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} has an invalid filing area: {value}."
                                    )
                                )

                            continue

        if not is_already_imported:
            if contract.name in (None, "") or (
                isinstance(contract.name, str)
                and re.sub(r"/\s\s+/", "", contract.name) == ""
            ):
                contract.name = slug

            if contract.valid_start_date is None:
                contract.valid_start_date = date(year=1970, month=1, day=1)

            if observed_issues_count != 0:
                self.partial_import_count += 1
                self.issue_count += observed_issues_count
            else:
                self.normal_import_count += 1

            if filing_area is not None:
                filing_area.save()

            for type_ in types:
                type_.save()

            # Save primary key first
            contract.save()

            contract.filing_area = filing_area
            contract.types.set(types)
            contract.save()
        else:
            self.already_imported_count += 1

    def import_contract_from_files(
        self, contract_root: str, files: list[str], valid_start_date: datetime
    ) -> None:
        contract = Contract(notes="")

        for file_ in files:
            with open(
                os.path.join(
                    contract_root,
                    file_,
                ),
                "r",
            ) as open_file:
                if file_ == "index.html":
                    metadata_failed = True

                    try:
                        metadata = self.parse_index(contract_root, open_file)
                    except ValueError as exc:
                        if self.verbosity >= 1:
                            self.stderr.write(
                                self.style.WARNING(
                                    f"Could not parse {contract_root} metadata: {exc}"
                                )
                            )

                        self.fatal_error_count += 1

                        return

                    self.assign_contract_metadata(
                        contract,
                        metadata,
                        os.path.basename(contract_root),
                    )
                elif file_.endswith(".pdf"):
                    # TODO
                    continue

    def import_all_contracts(self, git_dir) -> None:
        year_root = os.path.join(
            git_dir,
            "smlouvy",
        )

        for year_directory in sorted(os.listdir(year_root)):
            if int(year_directory) == 0:
                continue  # Out of range, TODO

            month_root = os.path.join(
                year_root,
                year_directory,
            )

            for month_directory in sorted(os.listdir(month_root)):
                day_root = os.path.join(
                    month_root,
                    month_directory,
                )

                for day_directory in sorted(os.listdir(day_root)):
                    contract_root = os.path.join(
                        git_dir,
                        "smlouvy",
                        year_directory,
                        month_directory,
                        day_directory,
                    )

                    for contract_directory in sorted(os.listdir(contract_root)):
                        this_contract_directory = os.path.join(
                            contract_root,
                            contract_directory,
                        )

                        if not os.path.isdir(this_contract_directory):
                            if self.verbosity >= 1:
                                self.stderr.write(
                                    self.style.WARNING(
                                        f"{this_contract_directory} is not a directory and thus invalid, skipping."
                                    )
                                )

                            self.fatal_error_count += 1

                            continue

                        valid_start_date = datetime(
                            year=int(year_directory),
                            month=int(month_directory),
                            day=int(day_directory),
                        )

                        self.import_contract_from_files(
                            this_contract_directory,
                            os.listdir(this_contract_directory),
                            valid_start_date,
                        )

        if self.verbosity >= 1:
            self.stdout.write(
                self.style.SUCCESS(
                    "\n"
                    f"Saved a total of {self.normal_import_count + self.partial_import_count} contracts.\n"
                    f"    {self.partial_import_count} contained a total of {self.issue_count} issues.\n"
                    f"    {self.already_imported_count} were already saved previously and skipped.\n"
                    f"    {self.fatal_error_count} potential contracts were unparseable."
                )
            )

    def handle(self, *args, **options) -> None:
        self.verbosity = options["verbosity"]

        git_dir = os.path.join(
            os.getcwd(),
            (options["directory"] if options["directory"] is not None else "git"),
        )

        if os.path.exists(git_dir):
            if not options["existing"]:
                if self.verbosity >= 1:
                    self.stderr.write(
                        self.style.ERROR(
                            f"Temporary git storage directory ({git_dir}) already exists.\n"
                            "As it could contain other data, it will not be removed.\n"
                            "Please remove it manually and try again or use the '--existing' "
                            "argument."
                        )
                    )

                return
            else:
                if self.verbosity >= 2:
                    self.stdout.write("Using existing git storage directory.")
        else:
            if self.verbosity >= 2:
                self.stdout.write("Cloning repository.")

            Repo.clone_from(
                options["repo_url"],
                git_dir,
                branch=options["branch"],
            )

            if self.verbosity >= 1:
                self.stdout.write(self.style.SUCCESS("Finished cloning repository."))

        if options["purge"]:
            Contract.objects.filter().delete()
            ContractType.objects.filter().delete()
            ContractFilingArea.objects.filter().delete()

            if self.verbosity >= 1:
                self.stdout.write(self.style.SUCCESS("Deleted all previous records."))

        if self.verbosity >= 2:
            self.stdout.write("\n")
        self.import_all_contracts(git_dir)

        if self.verbosity >= 1:
            self.stdout.write(self.style.SUCCESS("\nGit repository sync complete."))
