import io
import os

from datetime import date, datetime

import yaml

from ...models import Contract, ContractFilingArea, ContractType

from django.conf import settings
from django.core.management.base import BaseCommand
from git import Repo


class Command(BaseCommand):
    help = "Sync contract data from a remote Git repository."

    def add_arguments(self, parser):
        parser.add_argument(
            "repo_url",
            type=str,
            help="URL of the Git repository to clone",
        )
        parser.add_argument(
            "branch",
            type=str,
            help="Branch to use",
        )
        parser.add_argument(
            "--directory",
            type=str,
            help="Directory to store the cloned repository in",
        )
        parser.add_argument(
            "--existing",
            action="store_true",
            help="Use the existing storage directory, as long as it exists.",
        )

    def parse_index(
        self,
        contract_root: str,
        open_file,
    ) -> dict:
        split_contents = open_file.read().split("---")

        if len(split_contents) < 2:
            self.stderr.write(
                self.style.NOTICE(f"{contract_root} index does not have valid metadata.")
            )
            raise ValueError

        yaml_source = split_contents[1]

        try:
            parsed_metadata = yaml.safe_load(io.StringIO(yaml_source))
        except yaml.YAMLError as exc:
            self.stderr.write(
                self.style.NOTICE(f"Failed to parse {contract_root} metadata.")
            )

            raise ValueError from exc

        return parsed_metadata

    def assign_contract_metadata(
        self,
        contract: Contract,
        metadata: dict,
    ) -> None:
        filing_area = None
        types = []
        contract_already_exists = False

        for key, value in metadata.items():
            key = key.strip()

            if isinstance(value, str):
                value = value.strip()

            if key == "datum účinnosti" and isinstance(value, date):
                contract.valid_start_date = value
            elif key == "datum ukončení" and isinstance(value, date):
                contract.valid_end_date = value
            elif key == "title":
                contract.name = value

                if Contract.objects.filter(name=value).exists():
                    contract_already_exists = True
                    self.stdout.write(f"{contract.name} already exists.")

                    break
            elif key == "použité smluvní typy":
                if not isinstance(value, list):
                    continue

                for type_name in value:
                    if not isinstance(type_name, str):
                        continue

                    type_name = type_name.strip()

                    try:
                        type_instance = ContractType.objects.get(name=type_name)
                    except ContractType.DoesNotExist:
                        type_instance = ContractType(name=type_name)

                    types.append(type_instance)
            elif key == "předmět":
                contract.summary = value
            elif key == "stav":
                pass  # TODO
            elif key == "náklady":
                if isinstance(value, int):
                    if value <= 0:
                        continue

                    contract.cost_amount = value
                    contract.cost_unit = contract.CostUnits.TOTAL
                elif contract.name is not None:
                    self.stdout.write(
                        self.style.WARNING(f"Could not parse cost for contract {contract.name}.")
                    )
            elif key == "místo uložení":
                try:
                    filing_area = ContractFilingArea.objects.get(name=value)
                except ContractFilingArea.DoesNotExist:
                    if isinstance(value, str):
                        filing_area = ContractFilingArea(name=value)

        if not contract_already_exists:
            if filing_area is not None:
                filing_area.save()

            for type_ in types:
                type_.save()

            # Save primary key first
            contract.save()

            contract.filing_area = filing_area
            contract.types.set(types)
            contract.save()

    def import_contract_from_files(
        self,
        contract_root: str,
        files: list[str],
        valid_start_date: datetime
    ) -> None:
        for file_ in files:
            with open(
                os.path.join(
                    contract_root,
                    file_,
                ),
                "r"
            ) as open_file:
                contract = Contract()

                if file_ == "index.html":
                    try:
                        metadata = self.parse_index(contract_root, open_file)
                    except ValueError:
                        continue

                    if metadata is None:
                        continue

                    self.assign_contract_metadata(contract, metadata)
                elif file_.endswith(".pdf"):
                    continue

            contract.save()

    def import_all_contracts(self, git_dir) -> None:
        year_root = os.path.join(
            git_dir,
            "smlouvy",
        )

        saved_count = 0

        for year_directory in os.listdir(year_root):
            if int(year_directory) == 0:
                continue  # Out of range, TODO

            month_root = os.path.join(
                year_root,
                year_directory,
            )

            for month_directory in os.listdir(month_root):
                day_root = os.path.join(
                    month_root,
                    month_directory,
                )

                for day_directory in os.listdir(day_root):
                    contract_root = os.path.join(
                        git_dir,
                        "smlouvy",
                        year_directory,
                        month_directory,
                        day_directory,
                    )

                    for contract_directory in os.listdir(contract_root):
                        this_contract_directory = os.path.join(
                            contract_root,
                            contract_directory,
                        )

                        if not os.path.isdir(this_contract_directory):
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"{this_contract_directory} is not a directory, skipping."
                                )
                            )
                            continue

                        valid_start_date = datetime(
                            year=int(year_directory),
                            month=int(month_directory),
                            day=int(day_directory)
                        )

                        self.import_contract_from_files(
                            this_contract_directory,
                            os.listdir(this_contract_directory),
                            valid_start_date,
                        )
                        saved_count += 1

        self.stdout.write(
            self.style.SUCCESS(f"Saved {saved_count} contracts.")
        )

    def handle(self, *args, **options):
        git_dir = os.path.join(
            os.getcwd(),
            (
                options["directory"]
                if options["directory"] is not None
                else "git"
            )
        )

        if os.path.exists(git_dir):
            if not options["existing"]:
                self.stderr.write(
                    self.style.ERROR(
                        f"Temporary git storage directory ({git_dir}) already exists. "
                        "As it could contain other data, it will not be removed. Please "
                        "remove it manually and try again."
                    )
                )

                return
            else:
                self.stdout.write("Using existing git storage directory.")
        else:
            Repo.clone_from(
                options["repo_url"],
                git_dir,
                branch=options["branch"],
            )

            self.stdout.write("Cloning repository.")

        self.stdout.write("\n")
        self.import_all_contracts(git_dir)

        self.stdout.write(
            self.style.SUCCESS("\nGit repository sync complete.")
        )
