from django import forms
from django.core.exceptions import ValidationError
from webpack_loader.loader import WebpackLoader

from .models import Contract, ContracteeSignature, SigneeSignature


class ContractAdminForm(forms.ModelForm):
    def clean(self):
        clean_data = super().clean()

        if (
            not self.current_user.is_superuser
            and self.instance.is_approved
            is not clean_data.get("is_approved", self.instance.is_approved)
            and self.current_user == self.instance.created_by
        ):
            raise ValidationError(
                {
                    "is_approved": (
                        "Smlouva nemůže být schválena uživatelem, " "který ji nahrál."
                    )
                }
            )

        return clean_data

    class Media:
        js = (
            "shared/runtime.js",
            "shared/shared.js",
            "shared/admin_contract_form.js",
        )

    class Meta:
        widgets = {
            "summary": forms.Textarea(attrs={"rows": 2}),
        }


class ContractFileAdminForm(forms.ModelForm):
    class Media:
        js = ("shared/admin_contract_file_form.js",)

    class Meta:
        widgets = {
            "name": forms.TextInput(attrs={"list": "file-types"}),
        }


class SigneeAdminForm(forms.ModelForm):
    class Media:
        js = (
            "shared/runtime.js",
            "shared/shared.js",
            "shared/admin_signee_form.js",
        )

    def clean(self):
        cleaned_data = super().clean()

        if cleaned_data.get("entity_type") == "natural_person":
            if cleaned_data.get("ico_number") is not None:
                raise ValidationError(
                    {"ico_number": "IČO nesmí být pro fyzické osoby definováno."}
                )

            if cleaned_data.get("department") is not None:
                raise ValidationError(
                    {
                        "department": "Organizační složka nesmí být pro fyzické osoby definována."
                    }
                )

            if cleaned_data.get("date_of_birth") is None:
                raise ValidationError(
                    {
                        "date_of_birth": "Datum narození musí pro fyzické osoby být definováno."
                    }
                )

        if cleaned_data.get("entity_type") == "business_natural_person":
            if cleaned_data.get("ico_number") is None:
                raise ValidationError(
                    {
                        "ico_number": "IČO musí být pro podnikající fyzické osoby definováno."
                    }
                )

            if cleaned_data.get("department") is not None:
                raise ValidationError(
                    {
                        "department": "Organizační složka nesmí být pro podnikající fyzické osoby definována."
                    }
                )

            if cleaned_data.get("date_of_birth") is not None:
                raise ValidationError(
                    {
                        "date_of_birth": "Datum narození nesmí pro podnikající fyzické osoby být definováno."
                    }
                )

        if cleaned_data.get("entity_type") == "legal_entity":
            if cleaned_data.get("ico_number") is None:
                raise ValidationError(
                    {"ico_number": "IČO musí být pro právnické osoby definováno."}
                )

            if cleaned_data.get("date_of_birth") is not None:
                raise ValidationError(
                    {
                        "date_of_birth": "Datum narození nesmí pro právnické osoby být definováno."
                    }
                )

        return cleaned_data
