import io
import os
import string
from datetime import date, datetime

import yaml
from django.conf import settings
from django.core.management.base import BaseCommand

from git import Repo

from ...models import Contract, ContractFilingArea, ContractType


class Command(BaseCommand):
    help = "Sync contract data from a remote Git repository."

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.normal_import_count = 0
        self.partial_import_count = 0
        self.already_imported_count = 0
        self.issue_count = 0
        self.fatal_error_count = 0

    def add_arguments(self, parser):
        parser.add_argument(
            "repo_url",
            type=str,
            help="URL of the Git repository to clone",
        )
        parser.add_argument(
            "branch",
            type=str,
            help="Branch to use",
        )
        parser.add_argument(
            "--directory",
            type=str,
            help="Directory to store the cloned repository in",
        )
        parser.add_argument(
            "--existing",
            action="store_true",
            help="Use the existing storage directory, as long as it exists.",
        )
        parser.add_argument(
            "--purge",
            action="store_true",
            help="Purge all previous contracts before the import.",
        )

    def parse_index(
        self,
        contract_root: str,
        open_file,
    ) -> dict:
        split_contents = open_file.read().split("---")

        if len(split_contents) < 2:
            raise ValueError(f"{contract_root} index does not have valid metadata.")

        # Use Jan 1, 1970 as the default, if any defined year is 0.
        # Use Jan 1, 2100 as the default for infinitely valid contracts.
        split_contents[1] = (
            split_contents[1]
            .replace("0000-00-00", "1970-01-01")
            .replace("2222-22-22", "2100-01-01")
            .replace("2222-00-00", "2100-01-01")
            .replace("-32", "-31")
            .replace("\t", " ")
        )

        yaml_source = split_contents[1]

        try:
            parsed_metadata = yaml.safe_load(io.StringIO(yaml_source))
        except yaml.YAMLError as exc:
            raise ValueError(
                f"Failed to parse {contract_root} metadata: {exc}."
            ) from exc

        if parsed_metadata is None:
            raise ValueError(f"Got no metadata from {contract_root}.")

        return parsed_metadata

    def assign_contract_metadata(
        self,
        contract: Contract,
        metadata: dict,
        slug: str,
    ) -> None:
        filing_area = None
        types = []
        is_already_imported = False
        observed_issues_count = 0

        for key, value in metadata.items():
            key = key.strip()

            if isinstance(value, str):
                value = value.strip()

            match key:
                case "datum účinnosti":
                    if isinstance(value, date):
                        if date.year not in (1970, 2100):  # Ignore placeholder years
                            contract.valid_start_date = value
                    elif value is not None:
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadaný začátek platnosti: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has a broken valid start date: {value}."
                                )
                            )
                case "datum ukončení":
                    if isinstance(value, date):
                        if date.year not in (1970, 2100):  # Ignore placeholder years
                            contract.valid_end_date = value
                    elif value is not None and (
                        isinstance(value, str) and value.lower() != "na dobu neurčitou"
                    ):
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadaný konec platnosti: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has a broken valid end date: {value}."
                                )
                            )
                case "title":
                    contract.name = value

                    if Contract.objects.filter(name=value).exists():
                        value += f"(DUPLIKÁT - {slug})"

                    if Contract.objects.filter(name=value).exists():
                        is_already_imported = True
                        if self.verbosity >= 1:
                            self.stdout.write(f"{slug} already exists, skipping.")

                        break
                case "použité smluvní typy":
                    if isinstance(value, str):
                        value = string.capwords(value)

                        try:
                            type_instance = ContractType.objects.get(name=value)
                        except ContractType.DoesNotExist:
                            type_instance = ContractType(name=value)

                        types.append(type_instance)

                        continue
                    elif not isinstance(value, list):
                        observed_issues_count += 1
                        contract.notes += f"Špatně zadané typy: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} is missing types - not a list: {value}."
                                )
                            )

                        continue

                    for type_name in value:
                        if not isinstance(type_name, str):
                            observed_issues_count += 1
                            contract.notes += f"Nezaevidovaný typ: {type_name}\n"

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} is missing types - list item is not a string: {value}."
                                    )
                                )

                            continue

                        type_name = string.capwords(type_name.strip())

                        try:
                            type_instance = ContractType.objects.get(name=type_name)
                        except ContractType.DoesNotExist:
                            type_instance = ContractType(name=type_name)

                        types.append(type_instance)
                case "předmět":
                    contract.summary = value
                case "stav":
                    lower_value = value.lower()

                    if (
                        lower_value.startswith("splněn")
                        or lower_value.startswith("ukončen")
                        or lower_value.startswith("vypovězen")
                        or lower_value.startswith("odstoupen")
                        or lower_value in ("spněno",)
                    ):
                        contract.is_valid = False
                    elif (
                        lower_value.startswith("platn")
                        or lower_value.startswith("řešen")
                        or lower_value in ("v plnění", "v řešení")
                    ):
                        contract.is_valid = True
                    else:
                        observed_issues_count += 1
                        contract.notes += f"Neznámý stav: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Contract {slug} has an invalid state: {value}."
                                )
                            )
                case "náklady":
                    if isinstance(value, str):
                        formatted_value = value.replace(" ", "").replace(".00", ".0")

                        if formatted_value.isnumeric():
                            value = float(formatted_value)

                    if isinstance(value, (int, float)):
                        if value < 0:
                            observed_issues_count += 1
                            contract.notes += (
                                f"Původní, špatně zadané náklady: {value}\n"
                            )

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} has an invalid cost amount: {value}."
                                    )
                                )

                            continue

                        if value != 0:
                            contract.cost_amount = int(value)
                            contract.cost_unit = contract.CostUnits.TOTAL
                    elif isinstance(value, str) and value.endswith("Kč/h"):
                        split_value = value.split("Kč/h")

                        if (
                            len(split_value) != 2
                            or not split_value[0].strip().isnumeric()
                        ):
                            observed_issues_count += 1
                            contract.notes += f"Původní, neropoznané náklady: {value}\n"

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Could not parse cost for contract {slug}: {value}."
                                    )
                                )

                            continue

                        contract.cost_amount = int(split_value[0].strip())
                        contract.cost_unit = contract.CostUnits.HOUR
                    elif value not in (None, "0"):
                        observed_issues_count += 1
                        contract.notes += f"Původní, neropoznané náklady: {value}\n"

                        if self.verbosity >= 2:
                            self.stderr.write(
                                self.style.NOTICE(
                                    f"Could not parse cost for contract {slug}: {value}."
                                )
                            )
                case "místo uložení":
                    if isinstance(value, str):
                        value = string.capwords(value)  # Some normalization

                    try:
                        filing_area = ContractFilingArea.objects.get(name=value)
                    except ContractFilingArea.DoesNotExist:
                        if isinstance(value, str):
                            filing_area = ContractFilingArea(name=value)
                        else:
                            observed_issues_count += 1
                            contract.notes += f"Špatně zadaná spisovna: {value}\n"

                            if self.verbosity >= 2:
                                self.stderr.write(
                                    self.style.NOTICE(
                                        f"Contract {slug} has an invalid filing area: {value}."
                                    )
                                )

                            continue

        if not is_already_imported:
            if observed_issues_count != 0:
                self.partial_import_count += 1
                self.issue_count += observed_issues_count
            else:
                self.normal_import_count += 1

            if filing_area is not None:
                filing_area.save()

            for type_ in types:
                type_.save()

            # Save primary key first
            contract.save()

            contract.filing_area = filing_area
            contract.types.set(types)
            contract.save()
        else:
            self.already_imported_count += 1

    def import_contract_from_files(
        self, contract_root: str, files: list[str], valid_start_date: datetime
    ) -> None:
        contract = Contract(notes="")

        for file_ in files:
            with open(
                os.path.join(
                    contract_root,
                    file_,
                ),
                "r",
            ) as open_file:
                if file_ == "index.html":
                    metadata_failed = True

                    try:
                        metadata = self.parse_index(contract_root, open_file)
                    except ValueError as exc:
                        if self.verbosity >= 1:
                            self.stderr.write(
                                self.style.WARNING(
                                    f"Could not parse {contract_root} metadata: {exc}"
                                )
                            )

                        self.fatal_error_count += 1

                        continue

                    self.assign_contract_metadata(
                        contract,
                        metadata,
                        os.path.basename(contract_root),
                    )
                elif file_.endswith(".pdf"):
                    # TODO
                    continue

        contract.save()

    def import_all_contracts(self, git_dir) -> None:
        year_root = os.path.join(
            git_dir,
            "smlouvy",
        )

        for year_directory in sorted(os.listdir(year_root)):
            if int(year_directory) == 0:
                continue  # Out of range, TODO

            month_root = os.path.join(
                year_root,
                year_directory,
            )

            for month_directory in sorted(os.listdir(month_root)):
                day_root = os.path.join(
                    month_root,
                    month_directory,
                )

                for day_directory in sorted(os.listdir(day_root)):
                    contract_root = os.path.join(
                        git_dir,
                        "smlouvy",
                        year_directory,
                        month_directory,
                        day_directory,
                    )

                    for contract_directory in sorted(os.listdir(contract_root)):
                        this_contract_directory = os.path.join(
                            contract_root,
                            contract_directory,
                        )

                        if not os.path.isdir(this_contract_directory):
                            if self.verbosity >= 1:
                                self.stderr.write(
                                    self.style.WARNING(
                                        f"{this_contract_directory} is not a directory and thus invalid, skipping."
                                    )
                                )

                            self.fatal_error_count += 1

                            continue

                        valid_start_date = datetime(
                            year=int(year_directory),
                            month=int(month_directory),
                            day=int(day_directory),
                        )

                        self.import_contract_from_files(
                            this_contract_directory,
                            os.listdir(this_contract_directory),
                            valid_start_date,
                        )

        if self.verbosity >= 1:
            self.stdout.write(
                self.style.SUCCESS(
                    "\n"
                    f"Saved a total of {self.normal_import_count + self.partial_import_count} contracts.\n"
                    f"    {self.partial_import_count} contained a total of {self.issue_count} issues.\n"
                    f"    {self.already_imported_count} were already saved previously and skipped.\n"
                    f"    {self.fatal_error_count} potential contracts were unparseable."
                )
            )

    def handle(self, *args, **options) -> None:
        self.verbosity = options["verbosity"]

        git_dir = os.path.join(
            os.getcwd(),
            (options["directory"] if options["directory"] is not None else "git"),
        )

        if os.path.exists(git_dir):
            if not options["existing"]:
                if self.verbosity >= 1:
                    self.stderr.write(
                        self.style.ERROR(
                            f"Temporary git storage directory ({git_dir}) already exists.\n"
                            "As it could contain other data, it will not be removed.\n"
                            "Please remove it manually and try again or use the '--existing' "
                            "argument."
                        )
                    )

                return
            else:
                if self.verbosity >= 2:
                    self.stdout.write("Using existing git storage directory.")
        else:
            if self.verbosity >= 2:
                self.stdout.write("Cloning repository.")

            Repo.clone_from(
                options["repo_url"],
                git_dir,
                branch=options["branch"],
            )

            if self.verbosity >= 1:
                self.stdout.write(self.style.SUCCESS("Finished cloning repository."))

        if options["purge"]:
            Contract.objects.filter().delete()

            if self.verbosity >= 1:
                self.stdout.write(self.style.SUCCESS("Deleted all previous records."))

        if self.verbosity >= 2:
            self.stdout.write("\n")
        self.import_all_contracts(git_dir)

        if self.verbosity >= 1:
            self.stdout.write(self.style.SUCCESS("\nGit repository sync complete."))
