from django.contrib import admin

from shared.admin import MarkdownxGuardedModelAdmin

from .forms import (
    ContractAdminForm,
    SigneeAdminForm,
)
from .models import (
    Contract,
    Contractee,
    ContracteeRepresentative,
    ContracteeSignature,
    ContractFile,
    ContractFilingArea,
    ContractIntent,
    ContractIssue,
    ContractType,
    Signee,
    SigneeRepresentative,
    SigneeSignature,
)


class IndexHiddenModelAdmin(MarkdownxGuardedModelAdmin):
    def has_module_permission(self, request):
        return False


# BEGIN Contracts


class ContracteeSignatureInline(admin.TabularInline):
    model = ContracteeSignature
    extra = 0


class SigneeSignatureInline(admin.TabularInline):
    model = SigneeSignature
    extra = 0


class ContractFileInline(admin.TabularInline):
    model = ContractFile
    extra = 0


class ContractIntentInline(admin.TabularInline):
    model = ContractIntent
    extra = 0


class ContractAdmin(MarkdownxGuardedModelAdmin):
    form = ContractAdminForm

    fields = (
        "created_by",
        "types",
        "valid_start_date",
        "valid_end_date",
        "legal_state",
        "public_state",
        "paper_form_state",
        "publishing_rejection_comment",
        "tender_url",
        "identifier",
        "issues",
        "notes",
        "summary",
        "primary_contract",
        "agreement_url",
        "filing_area",
    )

    readonly_fields = ("created_by",)

    inlines = (
        ContracteeSignatureInline,
        SigneeSignatureInline,
        ContractFileInline,
        ContractIntentInline,
    )

    def save_model(self, request, obj, form, change) -> None:
        if obj.created_by is None:
            obj.created_by = request.user

        super().save_model(request, obj, form, change)


# END Contracts

# BEGIN Signing parties


class SigneeRepresentativeInline(admin.TabularInline):
    model = SigneeRepresentative
    extra = 0


class SigneeAdmin(MarkdownxGuardedModelAdmin):
    form = SigneeAdminForm

    inlines = (
        SigneeRepresentativeInline,
        SigneeSignatureInline,
    )


class ContracteeRepresentativeInline(admin.TabularInline):
    model = ContracteeRepresentative
    extra = 0


class ContracteeAdmin(MarkdownxGuardedModelAdmin):
    inlines = (ContracteeRepresentativeInline, ContracteeSignatureInline)


# END Signing parties


for model in (
    SigneeSignature,
    ContracteeSignature,
    SigneeRepresentative,
    ContracteeRepresentative,
    ContractType,
    ContractIntent,
):
    admin.site.register(model, IndexHiddenModelAdmin)

for model in (
    ContractIssue,
    ContractFilingArea,
):
    admin.site.register(model, MarkdownxGuardedModelAdmin)

admin.site.register(Signee, SigneeAdmin)
admin.site.register(Contractee, ContracteeAdmin)

admin.site.register(Contract, ContractAdmin)
