from django.contrib import admin
from django.utils.html import format_html

from shared.admin import MarkdownxGuardedModelAdmin

from .forms import ContractAdminForm, ContractFileAdminForm, SigneeAdminForm
from .models import (
    Contract,
    Contractee,
    ContracteeRepresentative,
    ContracteeSignature,
    ContractFile,
    ContractFilingArea,
    ContractIntent,
    ContractIssue,
    ContractType,
    Signee,
    SigneeRepresentative,
    SigneeSignature,
)


class IndexHiddenModelAdmin(MarkdownxGuardedModelAdmin):
    def has_module_permission(self, request):
        return False


# BEGIN Contracts


class ContractFileAdmin(IndexHiddenModelAdmin):
    form = ContractFileAdminForm


class ContracteeSignatureInline(admin.TabularInline):
    model = ContracteeSignature
    extra = 0


class SigneeSignatureInline(admin.TabularInline):
    model = SigneeSignature
    extra = 0


class ContractFileInline(admin.TabularInline):
    model = ContractFile
    form = ContractFileAdminForm
    extra = 0


class ContractIntentInline(admin.TabularInline):
    model = ContractIntent
    extra = 0


class ContractAdmin(MarkdownxGuardedModelAdmin):
    form = ContractAdminForm

    readonly_fields = ("created_by",)

    inlines = (
        ContracteeSignatureInline,
        SigneeSignatureInline,
        ContractFileInline,
        ContractIntentInline,
    )

    def get_fieldsets(self, request, obj=None):
        fieldsets = [
            (
                "Základní informace",
                {
                    "fields": [
                        "name",
                        "id_number",
                        "types",
                        "summary",
                        "public_state",
                        "publishing_rejection_comment",
                        "legal_state",
                        "primary_contract",
                    ]
                }
            ),
            (
                "Data",
                {
                    "fields": [
                        "valid_start_date",
                        "valid_end_date",
                    ]
                }
            ),
            (
                "Fyzický dokument",
                {
                    "fields": [
                        "paper_form_state",
                        "filing_area",
                    ]
                }
            ),
            (
                "Odkazy",
                {
                    "fields": [
                        "tender_url",
                        "agreement_url",
                    ]
                }
            ),
            (
                "Náklady",
                {
                    "fields": [
                        "cost_amount",
                        "cost_unit",
                    ]
                }
            ),
            (
                "Doplňující informace",
                {
                    "fields": [
                        "issues",
                        "notes",
                    ]
                }
            )
        ]

        if request.user.is_superuser or request.user.has_perm("approve", self):
            fieldsets[0][1]["fields"].insert(
                fieldsets[0][1]["fields"].index("primary_contract"),
                "approval_state",
            )

        return fieldsets

    def save_model(self, request, obj, form, change) -> None:
        if obj.created_by is None:
            obj.created_by = request.user

        super().save_model(request, obj, form, change)


# END Contracts

# BEGIN Signing parties


class SigneeRepresentativeInline(admin.TabularInline):
    model = SigneeRepresentative
    extra = 0


class SigneeAdmin(MarkdownxGuardedModelAdmin):
    form = SigneeAdminForm

    fields = (
        "name",
        "entity_type",
        "address_street_with_number",
        "address_district",
        "address_zip",
        "address_country",
        "ico_number",
        "load_ares_data_button",
        "date_of_birth",
        "department",
        "role",
    )

    readonly_fields = ("load_ares_data_button",)

    inlines = (SigneeRepresentativeInline,)

    def load_ares_data_button(self, obj):
        return format_html(
            "<button type=\"button\" id=\"load_ares_data\">Načíst data</button>"
        )

    load_ares_data_button.allow_tags = True
    load_ares_data_button.short_description = "ARES"


class ContracteeRepresentativeInline(admin.TabularInline):
    form = ContractFileAdminForm
    model = ContracteeRepresentative
    extra = 0


class ContracteeAdmin(MarkdownxGuardedModelAdmin):
    inlines = (ContracteeRepresentativeInline,)


# END Signing parties


for model in (
    SigneeSignature,
    ContracteeSignature,
    SigneeRepresentative,
    ContracteeRepresentative,
    ContractType,
    ContractIntent,
):
    admin.site.register(model, IndexHiddenModelAdmin)

admin.site.register(ContractFile, ContractFileAdmin)

for model in (
    ContractIssue,
    ContractFilingArea,
):
    admin.site.register(model, MarkdownxGuardedModelAdmin)

admin.site.register(Signee, SigneeAdmin)
admin.site.register(Contractee, ContracteeAdmin)

admin.site.register(Contract, ContractAdmin)
