"""
Django settings for registry project.

Generated by 'django-admin startproject' using Django 4.0.

For more information on this file, see
https://docs.djangoproject.com/en/4.0/topics/settings/

For the full list of settings and their values, see
https://docs.djangoproject.com/en/4.0/ref/settings/
"""

import os
import pathlib

import dj_database_url
import environ
import sentry_sdk
from sentry_sdk.integrations.django import DjangoIntegration

# Build paths inside the project like this: BASE_DIR / 'subdir'.
BASE_DIR = pathlib.Path(__file__).parents[2]


env = environ.Env()
environ.Env.read_env(os.path.join(BASE_DIR, ".env"))


# Quick-start development settings - unsuitable for production
# See https://docs.djangoproject.com/en/4.0/howto/deployment/checklist/

DEBUG = False

# SECURITY WARNING: keep the secret key used in production secret!
SECRET_KEY = env.str("SECRET_KEY")

ALLOWED_HOSTS = []

STATIC_ROOT = os.path.join(BASE_DIR, "staticfiles")
MEDIA_ROOT = os.path.join(BASE_DIR, "media")

SITE_URL = env.str("SITE_URL")


## Application definition

INSTALLED_APPS = [
    "admin_interface",
    "colorfield",
    "django_admin_index",
    "ordered_model",
    "django.contrib.admin",
    "django.contrib.auth",
    "django.contrib.contenttypes",
    "django.contrib.sessions",
    "django.contrib.messages",
    "django.contrib.staticfiles",
    "admin_auto_filters",
    "dbsettings",
    "nested_admin",
    "rangefilter",
    "guardian",
    "markdownx",
    "pirates",
    "webpack_loader",
    "media_server",
    "shared",
    "contracts",
    "oidc",
    "users",
]

MIDDLEWARE = [
    "django.middleware.security.SecurityMiddleware",
    "django.contrib.sessions.middleware.SessionMiddleware",
    "django.contrib.auth.middleware.AuthenticationMiddleware",
    "django.middleware.common.CommonMiddleware",
    "django.middleware.csrf.CsrfViewMiddleware",
    "django.contrib.messages.middleware.MessageMiddleware",
    "django.middleware.clickjacking.XFrameOptionsMiddleware",
    "django_http_exceptions.middleware.ExceptionHandlerMiddleware",
    "django_http_exceptions.middleware.ThreadLocalRequestMiddleware",
    "shared.middlewares.ClamAVMiddleware",
]

ROOT_URLCONF = "registry.urls"

TEMPLATES = [
    {
        "BACKEND": "django.template.backends.django.DjangoTemplates",
        "DIRS": [os.path.join(BASE_DIR, "registry", "templates")],
        "APP_DIRS": True,
        "OPTIONS": {
            "context_processors": [
                "django.template.context_processors.debug",
                "django.template.context_processors.request",
                "django.contrib.auth.context_processors.auth",
                "django.contrib.messages.context_processors.messages",
            ],
        },
    },
]

WSGI_APPLICATION = "registry.wsgi.application"


## Database
# https://docs.djangoproject.com/en/4.0/ref/settings/#databases

DATABASES = {"default": dj_database_url.config(conn_max_age=600)}

# Default primary key field type
# https://docs.djangoproject.com/en/4.0/ref/settings/#default-auto-field

DEFAULT_AUTO_FIELD = "django.db.models.BigAutoField"


## Authentication

# Password validation
# https://docs.djangoproject.com/en/4.1/ref/settings/#auth-password-validators

AUTH_PASSWORD_VALIDATORS = [
    {
        "NAME": "django.contrib.auth.password_validation.UserAttributeSimilarityValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation.MinimumLengthValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation.CommonPasswordValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation.NumericPasswordValidator",
    },
]
AUTH_USER_MODEL = "users.User"

AUTHENTICATION_BACKENDS = (
    "oidc.auth.RegistryOIDCAuthenticationBackend",
    "django.contrib.auth.backends.ModelBackend",
    "guardian.backends.ObjectPermissionBackend",
)

LOGIN_URL = "/"
LOGIN_REDIRECT_URL = "/"
LOGOUT_REDIRECT_URL = "/"

OIDC_RP_CLIENT_ID = env.str("OIDC_RP_CLIENT_ID")
OIDC_RP_CLIENT_SECRET = env.str("OIDC_RP_CLIENT_SECRET")
OIDC_RP_REALM_URL = env.str("OIDC_RP_REALM_URL")
OIDC_RP_SCOPES = "openid profile groups"
OIDC_RP_SIGN_ALGO = "RS256"
OIDC_RP_RESOURCE_ACCESS_CLIENT = env.str(
    "OIDC_RESOURCE_ACCESS_CLIENT", OIDC_RP_CLIENT_ID
)

OIDC_OP_JWKS_ENDPOINT = OIDC_RP_REALM_URL + "protocol/openid-connect/certs"
OIDC_OP_AUTHORIZATION_ENDPOINT = OIDC_RP_REALM_URL + "protocol/openid-connect/auth"
OIDC_OP_TOKEN_ENDPOINT = OIDC_RP_REALM_URL + "protocol/openid-connect/token"
OIDC_OP_USER_ENDPOINT = OIDC_RP_REALM_URL + "protocol/openid-connect/userinfo"

# Chobotnice - group source
CHOBOTNICE_API_URL = env.str("CHOBOTNICE_API_URL")


## Internationalization
# https://docs.djangoproject.com/en/4.0/topics/i18n/

LANGUAGE_CODE = "cs-cz"
TIME_ZONE = "Europe/Prague"
USE_I18N = True
USE_TZ = True


## Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/4.0/howto/static-files/

STATIC_URL = "/static/"

WEBPACK_LOADER = {
    "DEFAULT": {
        "CACHE": not DEBUG,
        "BUNDLE_DIR_NAME": "shared",
        "STATS_FILE": os.path.join(BASE_DIR, "webpack-stats.json"),
        "POLL_INTERVAL": 0.1,
        "IGNORE": [r".+\.hot-update.js", r".+\.map"],
    }
}


## Media files

MEDIA_URL = "/media/"

DEFAULT_FILE_STORAGE = "django_downloadview.storage.SignedFileSystemStorage"

## Server

USE_X_FORWARDED_HOST = True
SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")

X_FRAME_OPTIONS = "SAMEORIGIN"
SILENCED_SYSTEM_CHECKS = [
    "security.W019",
    "auth.W004",
]


## Admin

ADMIN_INDEX_AUTO_CREATE_APP_GROUP = True
ADMIN_INDEX_SHOW_REMAINING_APPS = True


ADMIN_ORDERING = {
    "contracts": [
        "Contract",
        "Contractee",
        "Signee",
        "ContractFilingArea",
        "ContractType",
        "ContractIssue",
    ],
}


# DBsettings

DBSETTINGS_VALUE_LENGTH = 65536


# ClamAV

CLAMD_USE_TCP = True
CLAMD_TCP_SOCKET = env.int("CLAMD_TCP_SOCKET")
CLAMD_TCP_ADDR = env.str("CLAMD_TCP_ADDR")


# Sentry

SENTRY_DSN = env.str("SENTRY_DSN", default="")

if SENTRY_DSN != "":
    sentry_sdk.init(
        dsn=SENTRY_DSN,
        integrations=[DjangoIntegration()],
        send_default_pii=True,
    )

## App-specific

NASTENKA_API_URL = "http://localhost:8009/contracts/api/notices"
NASTENKA_API_TOKEN = env.str("NASTENKA_API_TOKEN")

DEFAULT_CONTRACTEE_NAME = env.str("DEFAULT_CONTRACTEE_NAME")
DEFAULT_CONTRACTEE_STREET = env.str("DEFAULT_CONTRACTEE_STREET")
DEFAULT_CONTRACTEE_ZIP = env.str("DEFAULT_CONTRACTEE_ZIP")
DEFAULT_CONTRACTEE_DISTRICT = env.str("DEFAULT_CONTRACTEE_DISTRICT")
DEFAULT_COUNTRY = env.str("DEFAULT_COUNTRY")
DEFAULT_CONTRACTEE_ICO_NUMBER = env.str("DEFAULT_CONTRACTEE_ICO_NUMBER")

DEFAULT_STAFF_GROUPS = env.list("DEFAULT_STAFF_GROUPS")
