import React, { useState } from "react";
import classNames from "classnames";

import Button from "components/Button";
import { Card, CardActions, CardBody, CardHeadline } from "components/cards";
import ErrorMessage from "components/ErrorMessage";
import MarkdownEditor from "components/mde/MarkdownEditor";
import Modal from "components/modals/Modal";
import { urlRegex } from "utils";

const AnnouncementEditModal = ({
  announcement,
  onCancel,
  onConfirm,
  confirming,
  error,
  ...props
}) => {
  const [text, setText] = useState(announcement.content);
  const [link, setLink] = useState(announcement.link);
  const [linkValid, setLinkValid] = useState(null);
  const [noTextError, setNoTextError] = useState(false);

  const onTextInput = (newText) => {
    setText(newText);

    if (newText !== "") {
      setNoTextError(false);
    }
  };

  const onLinkInput = (newLink) => {
    setLink(newLink);

    if (!!newLink) {
      setLinkValid(urlRegex.test(newLink));
    }
  };

  const confirm = (evt) => {
    evt.preventDefault();

    let preventAction = false;
    const payload = {
      content: text,
    };

    if (!text) {
      setNoTextError(true);
      preventAction = true;
    }

    if (announcement.type === "voting" && !link) {
      setLinkValid(false);
      preventAction = true;
    } else {
      payload.link = link;
    }

    if (preventAction) {
      return;
    }

    onConfirm(payload);
  };

  return (
    <Modal containerClassName="max-w-lg" onRequestClose={onCancel} {...props}>
      <form onSubmit={confirm}>
        <Card>
          <CardBody>
            <div className="flex items-center justify-between mb-4">
              <CardHeadline>Upravit oznámení</CardHeadline>
              <button onClick={onCancel} type="button">
                <i className="ico--cross"></i>
              </button>
            </div>
            <MarkdownEditor
              value={text}
              onChange={onTextInput}
              error={
                noTextError
                  ? "Před úpravou oznámení nezapomeňte vyplnit jeho obsah."
                  : null
              }
              placeholder="Vyplňte text oznámení"
              toolbarCommands={[
                ["bold", "italic", "strikethrough"],
                ["link", "unordered-list", "ordered-list"],
              ]}
            />
            <div
              className={classNames("form-field mt-4", {
                hidden: announcement.type !== "voting",
                "form-field--error": linkValid === false,
              })}
            >
              <div className="form-field__wrapper form-field__wrapper--shadowed">
                <input
                  type="text"
                  className="text-input text-sm text-input--has-addon-l form-field__control"
                  value={link}
                  placeholder="URL hlasování"
                  onChange={(evt) => onLinkInput(evt.target.value)}
                />
                <div className="text-input-addon text-input-addon--l order-first">
                  <i className="ico--link"></i>
                </div>
              </div>
              {linkValid === false && (
                <div className="form-field__error">Zadejte platnou URL.</div>
              )}
            </div>
            {error && (
              <ErrorMessage className="mt-2">
                Při editaci došlo k problému: {error}
              </ErrorMessage>
            )}
          </CardBody>
          <CardActions right className="space-x-1">
            <Button
              hoverActive
              color="blue-300"
              className="text-sm"
              loading={confirming}
              type="submit"
            >
              Uložit
            </Button>
            <Button
              hoverActive
              color="red-600"
              className="text-sm"
              onClick={onCancel}
              type="button"
            >
              Zrušit
            </Button>
          </CardActions>
        </Card>
      </form>
    </Modal>
  );
};

export default AnnouncementEditModal;
