import { fabric } from "fabric";
import {
  clearObjects,
  sortObjects,
  transformTextLineBreaks,
  checkTextBoxHeight,
} from "../../components/canvas/utils";
import backgroundURL from "../../assets/template/urgent_text_banner/background.png";

let contractedByTextbox = null;

let textBox = null;
let backgroundImage = null;
let logoImage = null;
let previousLogoPosition = null;

const redraw = async (canvas, options) => {
  clearObjects([textBox, contractedByTextbox], canvas);

  const logoWidth = Math.ceil(canvas.width * 0.23);
  const logoSideMargin = Math.ceil(canvas.width * 0.07);
  const logoTopMargin = Math.ceil(canvas.height * 0.155);

  const contractedByTextSize = Math.ceil(canvas.height * 0.02);
  const contractedByTextMaxWidth = Math.ceil(canvas.width * 0.9);
  const contractedByTextSidesMargin = Math.ceil(canvas.width * 0.03);

  const textBoxWidth = Math.ceil(canvas.width * 0.65);
  const textSize = Math.ceil(canvas.height * 0.0725);
  const textLineHeight = 0.95;

  canvas.preserveObjectStacking = true;

  /* BEGIN Background render */

  if (backgroundImage === null) {
    backgroundImage = new Image();

    await new Promise((resolve) => {
      backgroundImage.onload = () => {
        resolve();
      };

      backgroundImage.src = backgroundURL;
    });

    backgroundImage = new fabric.Image(backgroundImage, {
      top: -20, // FIXME: Why???? Fabric.js, what are you trying to tell me?!
      left: -20,
      zIndex: 0,
      selectable: false,
    });
    backgroundImage.scaleToWidth(canvas.width + 22);

    canvas.add(backgroundImage);
  }

  /* END Background render */

  /* BEGIN Logo render */

  // A logo is provided, and it either hasn't been rendered yet or is a new one.
  const createNewLogo =
    (options.logoImage !== null &&
      (logoImage === null ||
        (options.logoImage !== null &&
          options.logoImage !== logoImage._element))) ||
    previousLogoPosition != options.logoPosition.id;

  previousLogoPosition = options.logoPosition.id;

  if (createNewLogo) {
    canvas.remove(logoImage);

    logoImage = new fabric.Image(options.logoImage, { selectable: false });
    logoImage.scaleToWidth(logoWidth);

    if (options.logoPosition.id == "top-right") {
      logoImage.set({
        left: canvas.width - logoWidth - logoSideMargin,
        top: logoTopMargin,
        zIndex: 11,
      });
    } else if (options.logoPosition.id == "top-left") {
      logoImage.set({
        left: logoSideMargin,
        top: logoTopMargin,
        zIndex: 11,
      });
    } else if (options.logoPosition.id == "bottom-left") {
      logoImage.set({
        left: logoSideMargin,
        top: canvas.height - logoTopMargin - logoImage.getScaledHeight(),
        zIndex: 11,
      });
    } else {
      logoImage.set({
        left: canvas.width - logoWidth - logoSideMargin,
        top: canvas.height - logoTopMargin - logoImage.getScaledHeight(),
        zIndex: 11,
      });
    }

    canvas.add(logoImage);
  }

  /* END Logo render */

  /* BEGIN Contracted by render */

  if (options.contractedBy !== null) {
    contractedByTextbox = new fabric.Textbox(options.contractedBy, {
      left:
        canvas.width - contractedByTextMaxWidth - contractedByTextSidesMargin,
      top: canvas.height - contractedByTextSidesMargin - contractedByTextSize,
      width: contractedByTextMaxWidth,
      fontFamily: "Roboto Condensed",
      fontSize: contractedByTextSize,
      textAlign: "right",
      fill: options.colors.contractedByText.value,
      selectable: false,
      zIndex: 10,
    });

    checkTextBoxHeight(contractedByTextbox, 1);

    canvas.add(contractedByTextbox);
  }

  /* END Contracted by render */

  /* BEGIN Text render */

  if (options.mainText !== null) {
    textBox = new fabric.Textbox(
      transformTextLineBreaks(
        options.mainText,
        textSize,
        "Bebas Neue",
        textBoxWidth,
      ),
      {
        width: textBoxWidth,
        left: (canvas.width - textBoxWidth) / 2,
        textAlign: "center",
        fontFamily: "Bebas Neue",
        fontSize: textSize,
        lineHeight: textLineHeight,
        fill: options.colors.text.value,
        selectable: false,
        zIndex: 10,
      },
    );

    checkTextBoxHeight(textBox, 5);

    canvas.add(textBox);

    textBox.top = (canvas.height - textBox.height) / 2;
  }

  /* END Text render */

  sortObjects(canvas);
};

export default redraw;
