import { fabric } from "fabric";
import {
  clearObjects,
  sortObjects,
  transformHighlightedText,
  checkTextBoxHeight,
} from "../../components/canvas/utils";
import COLORS from "../../colors";
import { PaddedHighlightingTextbox } from "../../components/canvas/textbox";

import bgImageSourceBlack from "../../assets/template/angle_event_right/bg_black.png";
import bgImageSourceWhite from "../../assets/template/angle_event_right/bg_white.png";

let mainImage = null;
let contractedByTextbox = null;
let previousColor = null;
let mainTextBox = null;

const redraw = async (canvas, options) => {
  clearObjects([contractedByTextbox, mainTextBox], canvas);

  const contractedByTextSize = Math.ceil(canvas.height * 0.035);
  const contractedByTextMaxWidth = Math.ceil(canvas.width * 0.9);
  const contractedByTextBottomMargin = Math.ceil(canvas.width * 0.02);
  const contractedByTextSideMargin = Math.ceil(canvas.width * 0.03);

  const textMarginLeft = Math.ceil(canvas.width * 0.52);
  const textMarginRight = Math.ceil(canvas.width * 0.1);

  let mainTextMarginTop = Math.ceil(canvas.height * 0.185);
  const mainTextBackgroundMarginTop = Math.ceil(canvas.height * 0.14);
  const mainTextSize = Math.ceil(canvas.height * 0.22);
  const mainTextHeightLimit = Math.ceil(mainTextSize * 2.2);
  const mainTextLineHeight = 0.85;

  canvas.preserveObjectStacking = true;

  /* BEGIN Main image render */

  if (previousColor !== options.colors.background || mainImage === null) {
    if (mainImage !== null) {
      canvas.remove(mainImage);
    }

    const image = new Image();

    const imageLoadPromise = new Promise((resolve) => {
      image.onload = () => {
        resolve();
      };

      if (options.colors.background.value === COLORS.black.value) {
        image.src = bgImageSourceBlack;
      } else {
        image.src = bgImageSourceWhite;
      }
    });
    await imageLoadPromise;

    mainImage = new fabric.Image(image, {
      left: 0,
      top: 0,
      zIndex: 0,
      selectable: false,
    });

    mainImage.scaleToWidth(canvas.width);

    canvas.add(mainImage);

    previousColor = options.colors.background;
  }

  /* END Main image render */

  if (options.mainText !== null) {
    /* BEGIN Name text render */

    const mainTextWidth = canvas.width - textMarginLeft - textMarginRight;

    const highlightedData = transformHighlightedText(
      options.mainText,
      mainTextSize,
      mainTextWidth,
      "Bebas Neue",
      options.colors.highlight.value,
      options.colors.highlightedText.value,
      { padWhenDiacritics: true },
    );

    mainTextBox = new PaddedHighlightingTextbox(highlightedData.text, {
      width: canvas.width,
      left: textMarginLeft,
      textAlign: "left",
      fontFamily: "Bebas Neue",
      fontSize: mainTextSize,
      lineHeight: mainTextLineHeight,
      fill: options.colors.mainText.value,
      styles: highlightedData.styles,
      selectable: false,
      highlightPadding: canvas.height * 0.003,
      zIndex: 10,
    });

    checkTextBoxHeight(mainTextBox, 3);

    canvas.add(mainTextBox);

    mainTextBox.top = mainTextMarginTop - highlightedData.paddingBottom;

    canvas.renderAll();

    /* END Main text render */
  }

  /* BEGIN Contracted by render */

  if (options.contractedBy !== null) {
    contractedByTextbox = new fabric.Textbox(options.contractedBy, {
      left: contractedByTextSideMargin,
      top: canvas.height - contractedByTextBottomMargin - contractedByTextSize,
      width: contractedByTextMaxWidth,
      fontFamily: "Roboto Condensed",
      fontSize: contractedByTextSize,
      textAlign: "left",
      fill: options.colors.contractedByText.value,
      selectable: false,
      zIndex: 10,
    });

    checkTextBoxHeight(contractedByTextbox, 1);

    canvas.add(contractedByTextbox);
  }

  /* END Contracted by render */

  sortObjects(canvas);
};

export default redraw;
