import { fabric } from "fabric";
import {
  clearObjects,
  sortObjects,
  transformHighlightedText,
  checkTextBoxHeight,
} from "../../components/canvas/utils";
import COLORS from "../../colors";
import { PaddedHighlightingTextbox } from "../../components/canvas/textbox";

import bgImageSource from "../../assets/template/social_cover_large_text/background.png";

let mainImage = null;
let contractedByTextbox = null;
let mainTextBox = null;

const redraw = async (canvas, options) => {
  clearObjects([contractedByTextbox, mainTextBox], canvas);

  const contractedByTextSize = Math.ceil(canvas.height * 0.035);
  const contractedByTextMaxWidth = Math.ceil(canvas.width * 0.9);
  const contractedByTextBottomMargin = Math.ceil(canvas.width * 0.02);
  const contractedByTextSideMargin = Math.ceil(canvas.width * 0.03);

  const textMarginLeft = Math.ceil(canvas.width * 0.15);
  const textWidth = Math.ceil(canvas.width * 0.7);
  const textMarginRight = Math.ceil(canvas.width * 0.23);

  let mainTextMarginBottom = Math.ceil(canvas.height * 0.17);
  const mainTextBackgroundMarginTop = Math.ceil(canvas.height * 0.14);
  const mainTextSize = Math.ceil(canvas.height * 0.26);
  const mainTextHeightLimit = Math.ceil(mainTextSize * 2.2);
  const mainTextLineHeight = 0.85;

  canvas.preserveObjectStacking = true;

  /* BEGIN Main image render */

  if (mainImage === null) {
    const image = new Image();

    console.log("Loading image");

    const imageLoadPromise = new Promise((resolve) => {
      image.onload = () => {
        resolve();
      };

      image.src = bgImageSource;
    });

    await imageLoadPromise;

    mainImage = new fabric.Image(image, {
      left: 0,
      top: 0,
      zIndex: 0,
      selectable: false,
    });

    mainImage.scaleToWidth(canvas.width);

    canvas.add(mainImage);
  }

  /* END Main image render */

  if (options.mainText !== null) {
    /* BEGIN Name text render */

    const highlightedData = transformHighlightedText(
      `*${options.mainText.replace(/\*/g, "")}*`,
      mainTextSize,
      textWidth,
      "Bebas Neue",
      "#000000",
      "#ffffff",
      { padWhenDiacritics: false },
    );

    if (Object.keys(highlightedData.styles).length == 2) {
      // Set the second line to yellow
      for (let i = 0; i < Object.keys(highlightedData.styles[1]).length; i++) {
        highlightedData.styles[1][i].fill =
          options.colors.highlightedText.value;
      }
    }

    mainTextBox = new PaddedHighlightingTextbox(highlightedData.text, {
      width: textWidth,
      left: textMarginLeft,
      textAlign: "center",
      fontFamily: "Bebas Neue",
      fontSize: mainTextSize,
      lineHeight: mainTextLineHeight,
      fill: options.colors.mainText.value,
      styles: highlightedData.styles,
      selectable: false,
      highlightPadding: canvas.height * 0.003,
      zIndex: 10,
    });

    checkTextBoxHeight(mainTextBox, 2);

    canvas.add(mainTextBox);

    let mainTextBoxTop =
      canvas.height - mainTextBox.height - mainTextMarginBottom;

    if (mainTextBox.textLines.length === 1) {
      mainTextBoxTop -= mainTextSize / 2;
    }

    mainTextBox.top = mainTextBoxTop - highlightedData.paddingBottom;

    canvas.renderAll();

    /* END Main text render */
  }

  /* BEGIN Contracted by render */

  if (options.contractedBy !== null) {
    contractedByTextbox = new fabric.Textbox(options.contractedBy, {
      left: contractedByTextSideMargin,
      top: canvas.height - contractedByTextBottomMargin - contractedByTextSize,
      width: contractedByTextMaxWidth,
      fontFamily: "Roboto Condensed",
      fontSize: contractedByTextSize,
      textAlign: "left",
      fill: options.colors.contractedByText.value,
      selectable: false,
      zIndex: 10,
    });

    checkTextBoxHeight(contractedByTextbox, 1);

    canvas.add(contractedByTextbox);
  }

  /* END Contracted by render */

  sortObjects(canvas);
};

export default redraw;
