package CF::Controller::Announcements;

use Mojo::Base 'Mojolicious::Controller';
use Mojo::Pg::PubSub;
use feature 'signatures';
no warnings qw{ experimental::signatures };

sub create ($c) {
    $c->openapi->valid_input or return;

    my $args   = $c->req->json;
    my $pubsub = Mojo::Pg::PubSub->new(pg => $c->pg);

    my $announcement = $c->schema->resultset('Announcement')->create({
        user_id  => $c->user->{id},
        type     => $args->{type} // 4,
        content  => $args->{content},
        link     => $args->{link},
    });

    # potrebujeme kvuli datumu
    $announcement = $c->schema->resultset('Announcement')->find({
        id => $announcement->id,
    });

    $pubsub->json('notify')->notify( notify => {
        event   => 'announcement_created',
        payload => $announcement->format(),
    });

    $c->render(
        status  => 201,
        openapi => { id => $announcement->id },
    );
};

sub list {
    my $c = shift->openapi->valid_input or return;

    my $args = $c->req->json;
    my ($cond, $attrs) = $c->search_parametrs( $args );

    $cond->{deleted} = undef;

    my $entries = $c->schema->resultset('Announcement')->search(
        $cond, $attrs
    );

    my @entries = ();
    my $count   = $c->schema->resultset('Announcement')->count($cond);

    ENTRY:
    while ( my $entry = $entries->next ) {
        push @entries, $c->spec_filter( $entry->format, 'Announcement');
    }

    $c->render(json => {
        data  => \@entries,
        total => $count,
    });
}

sub update ($c) {
    $c->openapi->valid_input or return;

    my $entry = $c->schema->resultset('Announcement')->find( $c->stash->{id} );
    return $c->error(404, 'Announcement entry not found') if ! $entry;

    my $update = $c->prepare_update_data( $entry, $c->req->json );
    my $pubsub = Mojo::Pg::PubSub->new(pg => $c->pg);
    my $guard  = $c->schema->txn_scope_guard;

    $entry->update( $update );

    $pubsub->json('notify')->notify( notify => {
        event   => 'announcement_changed',
        payload => {
            id => $entry->id,
            %{ $update },
        }
    });

    $guard->commit;

    $c->render(status => 204, text => '');
}

sub delete ($c) {
    $c->openapi->valid_input or return;

    my $entry = $c->schema->resultset('Announcement')->find( $c->stash->{id} );
    return $c->error(404, 'Announcement entry not found') if ! $entry;

    my $pubsub = Mojo::Pg::PubSub->new(pg => $c->pg);
    my $guard  = $c->schema->txn_scope_guard;

    $entry->update( { deleted => \'now()'} );

    $pubsub->json('notify')->notify( notify => {
        event   => 'announcement_deleted',
        payload => {
            id => $entry->id,
        }
    });

    $guard->commit;

    $c->render(status => 204, text => '');
}

1;
