package CF::Controller::Posts;

use Mojo::Base 'Mojolicious::Controller';
use Mojo::Pg::PubSub;
use feature 'signatures';
no warnings qw{ experimental::signatures };

sub create ($c) {
    $c->openapi->valid_input or return;

    my $args   = $c->req->json;
    my $pubsub = Mojo::Pg::PubSub->new(pg => $c->pg);

    # Navrh postupu muze predlozit jenom clen
    if ( $args->{type} == 0 && ! $c->user_roles->{member} ) {
        return $c->error(401, 'Insufficient permissions');
    }

    my $post = $c->schema->resultset('Post')->create({
        user_id => $c->user->{id},
        type    => $args->{type},
        content => $args->{content},
    });

    $pubsub->json('posts')->notify(
        posts => $post->view->format()
    );

    $c->render(
        status  => 201,
        openapi => { id => $post->id },
    );
};

sub get ($c) {
    $c->openapi->valid_input or return;

    my $post = $c->schema->resultset('Post_view')->find($c->stash->{id});
    return $c->error(404, 'Post not found') if ! $post;

    $c->render(openapi => $c->spec_filter($post->format(), 'Post'));
}

sub list ($c) {
    $c->openapi->valid_input or return;
    my $args = $c->validation->output;

    my ($cond, $attrs) = $c->search_parametrs( $args );

    if ( exists $args->{archived} && defined $args->{archived}) {
        $cond->{is_archived} = $args->{archived};
    }

    # omezeni zobrazovani nepotvrzenych navrhu postupu
    if ( $args->{type} ) {
        $cond->{type} = $args->{type};
        if ( $args->{type} == 0 && ! $c->user_roles->{chairman} ) {
            $cond->{state} = {'-in' => [1,2,3,4]}
        }
    }
    elsif (! $c->user_roles->{chairman}) {
        $cond->{-or} => [
            { type  => 1 },
            { state => { '-in' => [1,2,3,4] }},
        ];
    }

    my @posts = ();

    my $count = $c->schema->resultset('Post_view')->count($cond);

    if ( $count ) {
        my $posts = $c->schema->resultset('Post_view')->search($cond, $attrs);

        RECORD:
        while ( my $post = $posts->next() ) {
            push @posts, $c->spec_filter($post->format(), 'Post');
        }
    }

    $c->render(json => {
        data  => \@posts,
        total => $count,
    });
}

sub update ($c) {
    $c->openapi->valid_input or return;

    my $args   = $c->req->json;
    my $pubsub = Mojo::Pg::PubSub->new(pg => $c->pg);
    my $update = {};

    my $post = $c->schema->resultset('Post')->find($c->stash->{id});
    return $c->error(404, 'Post not found') if ! $post;

    if ( ! $c->user_roles->{chairman} ) {
        if ( $post->user_id != $c->user->{id} ) {
            return $c->error(403, 'Access deined');
        }
        else {
            delete $args->{is_archived};
            delete $args->{state};
        }
    }

    foreach my $key (qw(is_archoved state content)) {
        $update->{$key} = $args->{$key} if exists $args->{$key};
    }

    my $guard = $c->schema->txn_scope_guard;

    $post->add_to_history({
        user_id  => $c->user->{id},
        datetime => $post->datetime,
        content  => $post->content,
    });

    $post->update( $update );

    $pubsub->json('posts')->notify(
        posts => $post->view->format()
    );

    $guard->commit;

    $c->render(status => 204, text => '');
}

sub ranking ($c) {
    $c->openapi->valid_input or return;
    my $args = $c->validation->output;
    my $pubsub = Mojo::Pg::PubSub->new(pg => $c->pg);

    my $post = $c->schema->resultset('Post')->find($c->stash->{id});
    return $c->error(404, 'Post not found') if ! $post;

    my $user_ranking = $post->rankings({
        user_id => $c->user->{id},
    })->first;

    my $update = {
        ranking_likes    => $post->ranking_likes,
        ranking_dislikes => $post->ranking_dislikes,
    };

    my $user_ranking_update;

    if ($user_ranking && $user_ranking->ranking == $args->{ranking} ) {
        $user_ranking_update = 0;
    }
    else {
        $user_ranking_update = $args->{ranking};
    }

    if ($user_ranking && $user_ranking->ranking == 1 ) {
        $update->{ranking_likes}--;
        $update->{ranking_dislikes}++ if $args->{ranking} == -1;
    }
    elsif ($user_ranking && $user_ranking->ranking == -1 ) {
        $update->{ranking_dislikes}--;
        $update->{ranking_likes}++ if $args->{ranking} == 1;
    }
    else {
        $update->{ranking_likes}++ if $args->{ranking} == 1;
        $update->{ranking_dislikes}++ if $args->{ranking} == -1;
    }

    my $guard = $c->schema->txn_scope_guard;

    $post->update( $update );

    if ( $user_ranking ) {
        $user_ranking->update( { ranking => $user_ranking_update } );
    }
    else {
        $post->add_to_rankings({
            user_id => $c->user->{id},
            ranking => $user_ranking_update,
        });
    }

    $pubsub->json('posts')->notify(
        posts => $post->view->format()
    );
    $guard->commit;

    $c->render(status => 204, text => '');
}

sub ws {
    my $c = shift;

    $c->inactivity_timeout(300);

    my $pubsub = Mojo::Pg::PubSub->new(pg => $c->pg);

    $pubsub->listen(posts => sub($pubsub, $payload) {
        # FILTER?
        $c->send($payload);
    });

    $c->on(finish => sub ($c, $code, $reason = undef) {
        $pubsub->unlisten('streams');
        $c->app->log->debug("WebSocket closed with status $code");
    });
}

1;
